/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.actions;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import org.eclipse.gef.GraphicalViewer;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.ActionContributionItem;
import org.eclipse.jface.action.IMenuCreator;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.ui.actions.PartEventAction;

import de.bsvrz.buv.plugin.darstellung.internal.DarstellungMessages;
import de.bsvrz.buv.plugin.darstellung.model.Darstellung;
import de.bsvrz.buv.plugin.darstellung.util.DarstellungIcons;
import de.bsvrz.buv.plugin.dobj.model.DoTyp;
import de.bsvrz.buv.plugin.dobj.util.SelectionProperties;

/**
 * Stellt ein Dropdown-Menü als Selektionsfilter zur Verfügung. Nur die
 * ausgewählten Objekttypen werden bei der Selektion berücksichtigt. Nicht
 * ausgewählte Objekte werden bei der Selektion ignoriert.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 */
public class SelectionFilterAction extends PartEventAction implements IMenuCreator {

	private class DoTypAction extends Action {

		private final DoTyp doTyp;

		DoTypAction(final DoTyp doTyp) {
			super(doTyp.getName(), AS_CHECK_BOX);
			this.doTyp = doTyp;
			setChecked(getSelectionProperties().getMatchDoTypen().contains(doTyp));
		}

		@Override
		public void run() {
			if (isChecked()) {
				getSelectionProperties().getMatchDoTypen().add(doTyp);
			} else {
				getSelectionProperties().getMatchDoTypen().remove(doTyp);
			}
		}
	}

	private Menu menu;

	/**
	 * Initialisiert die Action.
	 */
	public SelectionFilterAction() {
		super(DarstellungMessages.SelectionFilter_Label);
		setImageDescriptor(DarstellungIcons.ActionSelectionFilter.getImageDescriptor());
		setToolTipText(DarstellungMessages.SelectionFilter_Tooltip);

		setId(getClass().toString());
		setActionDefinitionId(getClass().toString());

		setMenuCreator(this);
	}

	private void addActionToMenu(final Menu parent, final Action action) {
		final ActionContributionItem item = new ActionContributionItem(action);
		item.fill(parent, -1);
	}

	private Darstellung getDarstellung() {
		return getActivePart().getAdapter(Darstellung.class);
	}

	private SelectionProperties getSelectionProperties() {
		final GraphicalViewer viewer = getActivePart().getAdapter(GraphicalViewer.class);
		return (SelectionProperties) viewer.getProperty(SelectionProperties.class.toString());
	}

	@Override
	public Menu getMenu(final Menu parent) {
		return null;
	}

	@Override
	public Menu getMenu(final Control parent) {
		if (menu != null) {
			menu.dispose();
		}
		menu = new Menu(parent);

		final List<DoTyp> doTypenAlle = new ArrayList<>(getDarstellung().getDoTypen());

		final List<DoTyp> doTypen = new ArrayList<>();
		for (final DoTyp doTyp : doTypenAlle) {
			if (!doTypen.contains(doTyp)) {
				doTypen.add(doTyp);
			}
		}

		Collections.sort(doTypen, Comparator.comparing(DoTyp::getName));
		for (final DoTyp doTyp : doTypen) {
			addActionToMenu(menu, new DoTypAction(doTyp));
		}

		return menu;
	}

	@Override
	public void dispose() {
		if (menu != null) {
			menu.dispose();
			menu = null;
		}
	}

}
