/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.actions;

import java.util.List;

import org.eclipse.gef.GraphicalViewer;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.ISelectionService;
import org.eclipse.ui.IWorkbenchPart;

import com.bitctrl.lib.eclipse.viewer.ViewerTools;

import de.bsvrz.buv.plugin.darstellung.internal.DarstellungMessages;
import de.bsvrz.buv.plugin.darstellung.util.DarstellungIcons;
import de.bsvrz.buv.plugin.dobj.util.DobjUtil;
import de.bsvrz.dav.daf.main.config.SystemObject;

/**
 * Aktiviert und deaktiviert das Synchronisieren der Selektion zwischen
 * Workbench Parts.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class ToggleSynchronizeSelectionAction extends Action implements ISelectionChangedListener, ISelectionListener {

	private final IWorkbenchPart workbenchPart;

	/**
	 * Initialisiert die Aktion.
	 *
	 * @param part
	 *            der Workbench Part dessen Selektion mit anderen
	 *            synchronisieren werden soll.
	 */
	public ToggleSynchronizeSelectionAction(final IWorkbenchPart part) {
		super(DarstellungMessages.ToggleSynchronizeSelection_Label, AS_CHECK_BOX);
		workbenchPart = part;

		setImageDescriptor(DarstellungIcons.ActionSelectionLink.getImageDescriptor());
		setToolTipText(DarstellungMessages.ToggleSynchronizeSelection_Tooltip);

		setId(DarstellungActionFactory.TOGGLE_SYNCHRONIZE_SELECTION.getCommandId());
		setActionDefinitionId(DarstellungActionFactory.TOGGLE_SYNCHRONIZE_SELECTION.getCommandId());
	}

	@Override
	public void run() {
		final ISelectionService service = workbenchPart.getSite().getService(ISelectionService.class);
		if (isChecked()) {
			service.addSelectionListener(this);
			setChecked(true);
		} else {
			service.removeSelectionListener(this);
			setChecked(false);
		}
	}

	@Override
	public void selectionChanged(final SelectionChangedEvent event) {
		if (event.getSelectionProvider() != workbenchPart.getSite().getSelectionProvider()) {
			final GraphicalViewer viewer = workbenchPart.getAdapter(GraphicalViewer.class);
			viewer.setSelection(event.getSelection());
		}
	}

	@Override
	public void selectionChanged(final IWorkbenchPart part, final ISelection selection) {
		if (workbenchPart != part) {
			final GraphicalViewer viewer = workbenchPart.getAdapter(GraphicalViewer.class);
			final List<SystemObject> systemObjects = ViewerTools.getTypedSelection(selection, SystemObject.class);
			DobjUtil.setSelection(viewer, systemObjects);
		}
	}

}
