/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.commands;

import org.eclipse.core.runtime.Assert;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.gef.commands.Command;

import de.bsvrz.buv.plugin.darstellung.model.Ebene;
import de.bsvrz.buv.plugin.dobj.model.DoKomponente;
import de.bsvrz.buv.plugin.dobj.model.DoKompositum;
import de.bsvrz.buv.plugin.dobj.model.DoModel;

/**
 * Verschiebt ein Darstellungsobjekt von einer Ebene auf eine andere.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class AddDoModelToEbeneCommand extends Command {

	private final Ebene ebene;
	private final DoModel model;
	private EObject oldEbene;
	private int indexInOldEbene;

	public AddDoModelToEbeneCommand(final DoModel model, final Ebene ebene) {
		Assert.isNotNull(model, "model");
		Assert.isLegal(model.eContainer() != null, "model muss in einer Ebene oder einem Container liegen");
		Assert.isNotNull(ebene, "ebene");

		this.model = model;
		this.ebene = ebene;

		setLabel("Darstellungsobjekt auf andere Ebene verschieben");
	}

	@Override
	public void execute() {
		oldEbene = model.eContainer();
		if (oldEbene instanceof final Ebene ebene) {
			indexInOldEbene = ebene.getDoObjekte().indexOf(model);
		} else if (oldEbene instanceof DoKompositum<?>) {
			indexInOldEbene = ((DoKompositum<?>) oldEbene).getKomponenten().indexOf(model);
		} else {
			throw new IllegalArgumentException("Das DoObjekt muss sich in einer Ebene oder einem Kompositum befinden.");
		}
		redo();
	}

	@Override
	public void redo() {
		ebene.getDoObjekte().add(model);
	}

	@Override
	public void undo() {
		if (oldEbene instanceof final Ebene ebene) {
			ebene.getDoObjekte().add(indexInOldEbene, model);
		} else if (oldEbene instanceof DoKompositum) {
			((DoKompositum<DoKomponente>) oldEbene).getKomponenten().add(indexInOldEbene, model);
		}
	}

}
