/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.commands;

import org.eclipse.core.runtime.Assert;
import org.eclipse.gef.commands.Command;

import de.bsvrz.buv.plugin.darstellung.model.AutoEbene;
import de.bsvrz.buv.plugin.darstellung.model.Darstellung;
import de.bsvrz.buv.plugin.darstellung.util.DarstellungAdapter;
import de.bsvrz.buv.plugin.dobj.model.DoTyp;

/**
 * Fügt einer Autoebene einen Darstellungsobjekttyp hinzu.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class AddDoTypToAutoEbeneCommand extends Command {

	private final AutoEbene ebene;
	private DoTyp doTyp;
	private boolean useExistsDoTyp;

	public AddDoTypToAutoEbeneCommand(final DoTyp doTyp, final AutoEbene ebene) {
		Assert.isNotNull(doTyp, "doTyp");
		Assert.isNotNull(ebene, "ebene");

		this.doTyp = doTyp;
		this.ebene = ebene;

		setLabel("Darstellungsobjekttyp zur Autoebene hinzufügen");
	}

	@Override
	public boolean canExecute() {
		return !CommandUtil.contains(ebene.getDoTypen(), doTyp);
	}

	@Override
	public void execute() {
		Assert.isTrue(canExecute(), "command is not executable");

		doTyp = mergeDoTypAndCheckIfMustBeHandled(doTyp);

		redo();
	}

	private DoTyp mergeDoTypAndCheckIfMustBeHandled(final DoTyp dot) {
		final Darstellung darstellung = DarstellungAdapter.getDarstellung(ebene);

		final DoTyp existsDot = CommandUtil.getDoTyp(darstellung.getDoTypen(), dot);
		if (existsDot != null) {
			useExistsDoTyp = true;
			return existsDot;
		}

		useExistsDoTyp = false;
		return dot;
	}

	@Override
	public void redo() {
		ebene.getDoTypen().add(doTyp);

		if (!useExistsDoTyp) {
			final Darstellung darstellung = DarstellungAdapter.getDarstellung(ebene);
			darstellung.getDoTypen().add(doTyp);
		}
	}

	@Override
	public void undo() {
		if (!useExistsDoTyp) {
			final Darstellung darstellung = DarstellungAdapter.getDarstellung(ebene);
			darstellung.getDoTypen().remove(doTyp);
		}

		ebene.getDoTypen().remove(doTyp);
	}

}
