/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.commands;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Objects;

import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.util.EcoreUtil;

import de.bsvrz.buv.plugin.darstellung.model.AutoEbene;
import de.bsvrz.buv.plugin.darstellung.model.Darstellung;
import de.bsvrz.buv.plugin.darstellung.model.Ebene;
import de.bsvrz.buv.plugin.dobj.model.ConfiguratedDoTyp;
import de.bsvrz.buv.plugin.dobj.model.DoKomponente;
import de.bsvrz.buv.plugin.dobj.model.DoKompositum;
import de.bsvrz.buv.plugin.dobj.model.DoModel;
import de.bsvrz.buv.plugin.dobj.model.DoTyp;
import de.bsvrz.buv.plugin.doeditor.model.EditorDoTypReferenz;

public final class CommandUtil {

	/**
	 * Prüft, ob ein gegebener Darstellungsobjekttyp in einer Menge von
	 * Darstellungsobjekttypen enthalten ist.
	 */
	static boolean contains(final Collection<DoTyp> doTypen, final DoTyp doTyp) {
		return getDoTyp(doTypen, doTyp) != null;
	}

	/**
	 * Prüft, ob ein gegebener Darstellungsobjekttyp in einer Menge von
	 * Darstellungsobjekttypen enthalten ist.
	 *
	 * @return ist der Darstellungsobjekttyp in der Menge enthalten, wird der
	 *         entsprechende Darstellungsobjekttyp <em>aus der Menge</em>
	 *         zurückgegeben. Sonst <code>null</code>.
	 */
	static DoTyp getDoTyp(final Collection<DoTyp> doTypen, final DoTyp doTyp) {
		if (doTyp instanceof final ConfiguratedDoTyp modelDot) {
			for (final DoTyp dot : doTypen) {
				if (dot instanceof final ConfiguratedDoTyp darstellungDot) {
					if (modelDot.getId().equals(darstellungDot.getId())) {
						return darstellungDot;
					}
				}
			}
		} else if (doTyp instanceof final EditorDoTypReferenz modelDot) {
			for (final DoTyp dot : doTypen) {
				if (dot instanceof final EditorDoTypReferenz darstellungDot) {
					if (modelDot.getSystemObjekt().equals(darstellungDot.getSystemObjekt())) {
						return darstellungDot;
					}
				}
			}
		}

		// fallback
		return null;
	}

	/**
	 * Prüft ob es ein Darstellungsobjekt in einer Darstellung gibt, welches den
	 * angegebenen Darstellungsobjekttyp verwendet.
	 */
	static boolean isUsedDoTypInDarstellung(final DoTyp doTyp, final Darstellung darstellung,
			final AutoEbene excludeEbene) {
		return isUsedDoTypInDarstellung(doTyp, darstellung, new ArrayList<>(), excludeEbene);
	}

	/**
	 * Prüft ob es ein Darstellungsobjekt in einer Darstellung gibt, welches den
	 * angegebenen Darstellungsobjekttyp verwendet. Die Darstellungsobjekte in
	 * der Ausschlussliste werden dabei nicht mitgezählt.
	 */
	private static boolean isUsedDoTypInDarstellung(final DoTyp doTyp, final Darstellung darstellung,
			final Collection<DoModel> excludes, final AutoEbene excludeEbene) {

		final TreeIterator<EObject> allContents = EcoreUtil.getAllContents(darstellung, true);
		while (allContents.hasNext()) {
			final EObject next = allContents.next();
			if (!(next instanceof final Ebene ebene)) {
				continue;
			}
			if (ebene instanceof final AutoEbene autoEbene && ebene != excludeEbene) {
				for (final DoTyp dot : autoEbene.getDoTypen()) {
					if (Objects.equals(dot.getId(), doTyp.getId())) {
						return true;
					}
				}
			}

			for (final DoModel model : ebene.getDoObjekte()) {
				if (excludes.contains(model)) {
					continue;
				} else if (model.getDoTyp().getId() == doTyp.getId()) {
					return true;
				} else if (model instanceof DoKompositum<?>) {
					@SuppressWarnings("unchecked")
					final DoKompositum<DoModel> kompositum = (DoKompositum<DoModel>) model;
					if (isUsedDoTypInKompositum(doTyp, kompositum, excludes)) {
						return true;
					}
				}
			}
		}

		// fallback
		return false;
	}

	private static boolean isUsedDoTypInKompositum(final DoTyp doTyp, final DoKompositum<DoModel> kompositum,
			final Collection<DoModel> excludes) {
		for (final DoKomponente komponente : kompositum.getKomponenten()) {
			if (komponente instanceof final DoModel model) {
				if (excludes.contains(model)) {
					continue;
				} else if (model.getDoTyp() == doTyp) {
					return true;
				} else if (model instanceof DoKompositum<?>) {
					@SuppressWarnings("unchecked")
					final DoKompositum<DoModel> k = (DoKompositum<DoModel>) model;
					if (isUsedDoTypInKompositum(doTyp, k, excludes)) {
						return true;
					}
				}
			}
		}

		// fallback
		return false;
	}

	private CommandUtil() {
		// utility class
	}

}
