/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.commands;

import org.eclipse.core.runtime.Assert;
import org.eclipse.draw2d.geometry.Dimension;
import org.eclipse.gef.commands.Command;

import de.bsvrz.buv.plugin.darstellung.model.DarstellungFactory;
import de.bsvrz.buv.plugin.darstellung.model.DarstellungsSpalte;
import de.bsvrz.buv.plugin.darstellung.model.Ebene;
import de.bsvrz.buv.plugin.darstellung.model.StilisierteDarstellung;
import de.bsvrz.buv.plugin.dobj.model.DoModel;

/**
 * Legt eine neue Darstellungsspalte in einer stilisierten Darstellung an.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class CreateDarstellungsSpalteCommand extends Command {

	private final StilisierteDarstellung darstellung;
	private DarstellungsSpalte spalte;

	public CreateDarstellungsSpalteCommand(final StilisierteDarstellung darstellung) {
		Assert.isNotNull(darstellung, "darstellung");

		this.darstellung = darstellung;
	}

	@Override
	public void execute() {
		spalte = createDarstellungsspalte();

		redo();
	}

	private DarstellungsSpalte createDarstellungsspalte() {
		final DarstellungsSpalte ds = DarstellungFactory.eINSTANCE.createDarstellungsSpalte();
		String name = "Spalte";
		if (!isErsteSpalte()) {
			name = darstellung.getSpalten().size() + 1 + ". " + name;
		}
		ds.setName(name);
		return ds;
	}

	@Override
	public void redo() {
		if (isErsteSpalte()) {
			addObjekteToSpalte();
		} else {
			vergroessereDarstellungUmSpaltenBreite();
		}

		addDarstellungsSpalteToDarstellung();
	}

	private boolean isErsteSpalte() {
		return darstellung.getSpalten().isEmpty();
	}

	private void addDarstellungsSpalteToDarstellung() {
		darstellung.getSpalten().add(spalte);
	}

	private void addObjekteToSpalte() {
		spalte.setWidth(darstellung.getSize().width);
		for (final Ebene ebene : darstellung.getEbenen()) {
			for (final DoModel doModel : ebene.getDoObjekte()) {
				spalte.getDoObjekte().add(doModel);
			}
		}
	}

	private void vergroessereDarstellungUmSpaltenBreite() {
		final Dimension size = darstellung.getSize();
		darstellung.setSize(new Dimension(size.width + spalte.getWidth(), size.height));
	}

	@Override
	public void undo() {
		removeDarstellungsSpalteFromDarstellung();
		if (darstellung.getSpalten().isEmpty()) {
			removeObjekteFromSpalte();
		} else {
			verkleinereDarstellungUmSpaltenBreite();
		}
	}

	private void removeDarstellungsSpalteFromDarstellung() {
		darstellung.getSpalten().remove(spalte);
	}

	private void removeObjekteFromSpalte() {
		for (final Ebene ebene : darstellung.getEbenen()) {
			for (final DoModel doModel : ebene.getDoObjekte()) {
				spalte.getDoObjekte().remove(doModel);
			}
		}
	}

	private void verkleinereDarstellungUmSpaltenBreite() {
		final Dimension size = darstellung.getSize();
		darstellung.setSize(new Dimension(size.width - spalte.getWidth(), size.height));

	}

}
