/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.commands;

import org.eclipse.core.runtime.Assert;
import org.eclipse.draw2d.geometry.Dimension;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.gef.commands.Command;

import com.bitctrl.lib.eclipse.emf.gef.model.Sized;

import de.bsvrz.buv.plugin.darstellung.model.Darstellung;
import de.bsvrz.buv.plugin.darstellung.model.DarstellungsSpalte;
import de.bsvrz.buv.plugin.darstellung.model.Ebene;
import de.bsvrz.buv.plugin.darstellung.model.StilisierteDarstellung;
import de.bsvrz.buv.plugin.darstellung.util.DarstellungAdapter;
import de.bsvrz.buv.plugin.dobj.model.DoModel;

/**
 * Legt ein neues Darstellungsobjekts in einer Ebene oder Darstellungsspalte an.
 *
 * <p>
 * <em>Hinweis:</em> Die Position und ggf. Größe des neuen Darstellungsobjekts
 * muss mit {@link #setLocation(Point)} angegeben werden, wenn im Modell keine
 * Position angegeben ist.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 * @see #setLocation(Point)
 */
public class CreateDoModelCommand extends Command {

	private final DoModel model;
	private final Ebene ebene;
	private final DarstellungsSpalte darstellungsSpalte;
	private Point location;
	private Dimension size;

	public CreateDoModelCommand(final DoModel model, final Ebene ebene) {
		this(model, ebene, null);
	}

	public CreateDoModelCommand(final DoModel model, final Ebene ebene, final DarstellungsSpalte darstellungsSpalte) {
		Assert.isNotNull(model, "model");
		Assert.isNotNull(ebene, "ebene");
		assertDarstellungsSpalteNeeded(ebene, darstellungsSpalte);

		this.model = model;
		this.ebene = ebene;
		this.darstellungsSpalte = darstellungsSpalte;

		setLabel("Darstellungsonjekt anlegen");
	}

	private void assertDarstellungsSpalteNeeded(final Ebene e, final DarstellungsSpalte s) {
		final Darstellung darstellung = DarstellungAdapter.getDarstellung(e);
		if (darstellung instanceof final StilisierteDarstellung stilisierteDarstellung) {
			if (!stilisierteDarstellung.getSpalten().isEmpty()) {
				Assert.isNotNull(s, "darstellungsSpalte");
			}
		}
	}

	public Point getLocation() {
		return location;
	}

	public void setLocation(final Point location) {
		this.location = location;
	}

	public Dimension getSize() {
		return size;
	}

	public void setSize(final Dimension size) {
		this.size = size;
	}

	@Override
	public boolean canExecute() {
		return location != null || model.getLocation() != null;
	}

	@Override
	public void execute() {
		Assert.isTrue(canExecute(), "command is not executable");

		applyConstraintsIfNecessary();

		redo();
	}

	private void applyConstraintsIfNecessary() {
		if (getLocation() != null) {
			model.setLocation(getLocation());
		} else {
			Assert.isNotNull(model.getLocation(), "entweder constraint oder location muss gesetzt sein");
		}
		if (model instanceof Sized && getSize() != null) {
			((Sized) model).setSize(getSize());
		}
	}

	@Override
	public void redo() {
		ebene.getDoObjekte().add(model);
		if (darstellungsSpalte != null) {
			darstellungsSpalte.getDoObjekte().add(model);
		}
	}

	@Override
	public void undo() {
		if (darstellungsSpalte != null) {
			darstellungsSpalte.getDoObjekte().remove(model);
		}
		ebene.getDoObjekte().remove(model);
	}

}
