/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.commands;

import org.eclipse.core.runtime.Assert;
import org.eclipse.gef.commands.Command;

import de.bsvrz.buv.plugin.darstellung.model.Darstellung;
import de.bsvrz.buv.plugin.darstellung.model.DarstellungsSpalte;
import de.bsvrz.buv.plugin.darstellung.model.Ebene;
import de.bsvrz.buv.plugin.darstellung.model.StilisierteDarstellung;
import de.bsvrz.buv.plugin.darstellung.util.DarstellungAdapter;
import de.bsvrz.buv.plugin.dobj.model.DoModel;

/**
 * Löscht ein Darstellungsobjekt aus einer Darstellung.
 *
 * <p>
 * Das Darstellungsobjekt wird aus der Ebene und der Darstellungsspalte, falls
 * vorhanden, entfernt.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class DeleteDoModelCommand extends Command {

	private final DoModel model;
	private Ebene ebene;
	private int modelIndexInEbenen;
	private DarstellungsSpalte darstellungsSpalte;
	private int modelIndexInDarstellungsSpalte;

	public DeleteDoModelCommand(final DoModel model) {
		Assert.isNotNull(model, "model");

		this.model = model;

		setLabel("Darstellungsbjekt löschen");
	}

	@Override
	public boolean canExecute() {
		return model.eContainer() instanceof Ebene;
	}

	@Override
	public void execute() {
		Assert.isTrue(canExecute(), "command is not executable");

		ebene = getEbene();
		modelIndexInEbenen = getEbenenIndex(ebene);
		final Darstellung darstellung = getDarstellung(ebene);
		darstellungsSpalte = getDarstellungsspalteIfMustBeHandled(darstellung);
		modelIndexInDarstellungsSpalte = getModelIndexInDarstellungsSpalteIfMustBeHandled(darstellungsSpalte);

		redo();
	}

	private int getModelIndexInDarstellungsSpalteIfMustBeHandled(final DarstellungsSpalte spalte) {
		if (spalte != null) {
			return spalte.getDoObjekte().indexOf(model);
		}

		// fallback
		return -1;
	}

	private int getEbenenIndex(final Ebene e) {
		return e.getDoObjekte().indexOf(model);
	}

	private Darstellung getDarstellung(final Ebene e) {
		final Darstellung d = DarstellungAdapter.getDarstellung(e);
		Assert.isNotNull(d, "darstellung");
		return d;
	}

	private Ebene getEbene() {
		final Ebene e = (Ebene) model.eContainer();
		Assert.isNotNull(e, "ebene");
		return e;
	}

	private DarstellungsSpalte getDarstellungsspalteIfMustBeHandled(final Darstellung d) {
		if (d instanceof StilisierteDarstellung) {
			for (final DarstellungsSpalte e : ((StilisierteDarstellung) d).getSpalten()) {
				if (e.getDoObjekte().contains(model)) {
					return e;
				}
			}
		}

		// fallback
		return null;
	}

	@Override
	public void redo() {
		removeModelFromDarstellungsSpalteIfNecessary();
		removeModelFromEbene();
	}

	private void removeModelFromDarstellungsSpalteIfNecessary() {
		if (darstellungsSpalte != null) {
			darstellungsSpalte.getDoObjekte().remove(modelIndexInDarstellungsSpalte);
		}
	}

	private void removeModelFromEbene() {
		ebene.getDoObjekte().remove(modelIndexInEbenen);
	}

	@Override
	public void undo() {
		addModelToEbene();
		addModelToDarstellungsSpalteIfNecessary();
	}

	private void addModelToEbene() {
		ebene.getDoObjekte().add(modelIndexInEbenen, model);
	}

	private void addModelToDarstellungsSpalteIfNecessary() {
		if (darstellungsSpalte != null) {
			darstellungsSpalte.getDoObjekte().add(modelIndexInDarstellungsSpalte, model);
		}
	}

}
