/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.commands;

import org.eclipse.core.runtime.Assert;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.gef.commands.Command;
import org.eclipse.gef.commands.CompoundCommand;

import de.bsvrz.buv.plugin.darstellung.model.AutoEbene;
import de.bsvrz.buv.plugin.darstellung.model.Darstellung;
import de.bsvrz.buv.plugin.darstellung.model.Ebene;
import de.bsvrz.buv.plugin.dobj.model.DoModel;
import de.bsvrz.buv.plugin.dobj.model.DoTyp;

/**
 * Löscht eine Ebene aus einer Darstellung.
 *
 * <p>
 * Löscht auch die Darstellungsobjekte der zu löschenden Ebene aus der
 * Darstellung.
 *
 * <p>
 * Eine Darstellung muss mindestens eine Ebene besitze, also kann eine Ebene nur
 * gelöscht werden, wenn es noch mindestens eine andere gibt.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class DeleteEbeneCommand extends CompoundCommand {

	private final Ebene ebene;
	private final EObject parent;

	public DeleteEbeneCommand(final Ebene ebene) {
		Assert.isNotNull(ebene, "ebene");

		this.ebene = ebene;

		parent = ebene.eContainer();
		Assert.isNotNull(parent, "darstellung/ebene");

		setLabel("Ebene löschen");
	}

	@Override
	public boolean canExecute() {
		if (parent instanceof Ebene) {
			return ((Ebene) parent).getEbenen().size() > 0;
		} else if (parent instanceof Darstellung) {
			return ((Darstellung) parent).getEbenen().size() > 1;
		}
		return false;
	}

	@Override
	public void execute() {
		Assert.isTrue(canExecute(), "command is not executable");

		for (final DoModel model : ebene.getDoObjekte()) {
			add(new DeleteDoModelCommand(model));
		}

		if (ebene instanceof final AutoEbene autoEbene) {
			for (final DoTyp doTyp : autoEbene.getDoTypen()) {
				add(new RemoveDoTypFromAutoEbeneCommand(doTyp, autoEbene));
			}
		}

		add(new InternalDeleteEbeneCommand());

		super.execute();
	}

	private class InternalDeleteEbeneCommand extends Command {

		private int index;

		@Override
		public void execute() {
			if (parent instanceof Ebene) {
				index = ((Ebene) parent).getEbenen().indexOf(ebene);
			} else if (parent instanceof Darstellung) {
				index = ((Darstellung) parent).getEbenen().indexOf(ebene);
			}
			redo();
		}

		@Override
		public void redo() {
			if (parent instanceof Ebene) {
				((Ebene) parent).getEbenen().remove(index);
			} else if (parent instanceof Darstellung) {
				((Darstellung) parent).getEbenen().remove(index);
			}
		}

		@Override
		public void undo() {
			if (parent instanceof Ebene) {
				((Ebene) parent).getEbenen().add(index, ebene);
			} else if (parent instanceof Darstellung) {
				((Darstellung) parent).getEbenen().add(index, ebene);
			}
		}

	}

}
