/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.commands;

import org.eclipse.core.runtime.Assert;
import org.eclipse.gef.commands.Command;

import de.bsvrz.buv.plugin.darstellung.model.AutoEbene;
import de.bsvrz.buv.plugin.darstellung.model.Darstellung;
import de.bsvrz.buv.plugin.darstellung.util.DarstellungAdapter;
import de.bsvrz.buv.plugin.dobj.model.DoTyp;

/**
 * Entfernt einen Darstellungsobjekttyp aus einer Autoebene.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class RemoveDoTypFromAutoEbeneCommand extends Command {

	private final AutoEbene ebene;
	private final DoTyp doTyp;
	private int doTypIndexInEbene;
	private boolean removeDoTypFromDarstellung;

	public RemoveDoTypFromAutoEbeneCommand(final DoTyp doTyp, final AutoEbene ebene) {
		Assert.isNotNull(doTyp, "doTyp");
		Assert.isNotNull(ebene, "ebene");

		this.doTyp = doTyp;
		this.ebene = ebene;

		setLabel("Darstellungsobjekttyp aus Autoebene entfernen");
	}

	@Override
	public void execute() {
		removeDoTypFromDarstellung = !isNeededDoTyp();
		doTypIndexInEbene = getIndexInEbene(ebene);

		redo();
	}

	private int getIndexInEbene(final AutoEbene e) {
		return e.getDoTypen().indexOf(doTyp);
	}

	private boolean isNeededDoTyp() {
		final Darstellung darstellung = DarstellungAdapter.getDarstellung(ebene);
		return CommandUtil.isUsedDoTypInDarstellung(doTyp, darstellung, ebene);
	}

	@Override
	public void redo() {
		ebene.getDoTypen().remove(doTypIndexInEbene);

		if (removeDoTypFromDarstellung) {
			final Darstellung darstellung = DarstellungAdapter.getDarstellung(ebene);
			darstellung.getDoTypen().remove(doTyp);
		}
	}

	@Override
	public void undo() {
		ebene.getDoTypen().add(doTypIndexInEbene, doTyp);

		if (removeDoTypFromDarstellung) {
			final Darstellung darstellung = DarstellungAdapter.getDarstellung(ebene);
			darstellung.getDoTypen().add(doTyp);
		}
	}

}
