/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.decorator;

import java.util.List;

import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.IDecoration;
import org.eclipse.jface.viewers.ILightweightLabelDecorator;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.PlatformUI;

import com.bitctrl.lib.eclipse.emf.eclipse.model.Named;

import de.bsvrz.buv.plugin.darstellung.util.DarstellungValidator;
import de.bsvrz.buv.plugin.dobj.internal.DObjPlugin;

/**
 * Der Dekorierer markiert nicht lesbare Einstellungen in JFace-Viewern.
 *
 * <p>
 * Eine nicht lesbare Einstellung wird
 * <ul>
 * <li>in grauer Schrift angezeigt,</li>
 * <li>mit einem roten Kreuz unten links im Icon markiert und</li>
 * <li>hat den Suffix " (Fehler beim Lesen)"</li>
 * </ul>
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class EmfEinstellungenDecorator extends LabelProvider implements ILightweightLabelDecorator {

	private final ImageDescriptor errorImageDescriptor = DObjPlugin.getDefault().getImageDescriptor("icons/error.gif");
	private final ImageDescriptor warningImageDescriptor = DObjPlugin.getDefault()
			.getImageDescriptor("icons/warning.gif");
	private Color errorColor;

	{
		Assert.isNotNull(errorImageDescriptor);

		final Display display = PlatformUI.getWorkbench().getDisplay();
		display.syncExec(() -> {
			errorColor = display.getSystemColor(SWT.COLOR_GRAY);
			Assert.isNotNull(errorColor);
		});
	}

	@Override
	public void decorate(final Object element, final IDecoration decoration) {
		if (!(element instanceof Named)) {
			return;
		}

		final List<IStatus> status = DarstellungValidator.validiere((Named) element);

		if (status.isEmpty() || !status.stream().filter(s -> !Status.OK_STATUS.equals(s)).findAny().isPresent()) {
			return;
		}

		if (status.stream().filter(s -> s.getSeverity() == IStatus.ERROR).findAny().isPresent()) {
			decoration.addSuffix(" (Fehler beim Lesen)");
			decoration.addOverlay(errorImageDescriptor, IDecoration.BOTTOM_LEFT);
			decoration.setForegroundColor(errorColor);
		} else if (status.stream().filter(s -> s.getSeverity() == IStatus.WARNING).findAny().isPresent()) {
			decoration.addSuffix(" (Warnung beim Lesen)");
			decoration.addOverlay(warningImageDescriptor, IDecoration.BOTTOM_LEFT);
		}

	}

}
