/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.dialogs;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.eclipse.jface.dialogs.TitleAreaDialog;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.layout.TableColumnLayout;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.CheckboxCellEditor;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.EditingSupport;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.viewers.ViewerComparator;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Shell;

import de.bsvrz.buv.plugin.darstellung.model.Ansicht;
import de.bsvrz.buv.plugin.darstellung.model.Ausschnitt;

public class AusschnitteVerwaltenDialog extends TitleAreaDialog {

	private final Map<Ausschnitt, String> ausschnitte = new HashMap<>();
	private final Ansicht ansicht;
	private Ausschnitt defaultAusschnitt;

	public final Ausschnitt getDefaultAusschnitt() {
		return defaultAusschnitt;
	}

	private Button removeButton;

	public AusschnitteVerwaltenDialog(final Shell shell, final Ansicht ansicht) {
		super(shell);
		this.ansicht = ansicht;
		defaultAusschnitt = ansicht.getAusschnitt();
		for (final Ausschnitt ausschnitt : ansicht.getAusschnitte()) {
			ausschnitte.put(ausschnitt, ausschnitt.getName());
		}
	}

	@Override
	protected Control createDialogArea(final Composite parent) {

		setTitle(ansicht.getName() + " (Ausschnitte)");
		setMessage("Bearbeiten oder löschen Sie die gewünschten Ausschnitte!");

		final Composite panel = new Composite(parent, SWT.None);
		GridDataFactory.fillDefaults().grab(true, true).applyTo(panel);
		GridLayoutFactory.swtDefaults().numColumns(1).applyTo(panel);

		final Composite tablePanel = new Composite(panel, SWT.NONE);
		GridDataFactory.fillDefaults().grab(true, true).applyTo(tablePanel);

		final TableColumnLayout tabLayout = new TableColumnLayout();
		tablePanel.setLayout(tabLayout);

		final TableViewer viewer = new TableViewer(tablePanel, SWT.FULL_SELECTION | SWT.SINGLE);
		viewer.getTable().setHeaderVisible(true);
		viewer.getTable().setLinesVisible(true);

		viewer.setContentProvider(new ArrayContentProvider());
		viewer.setComparator(new ViewerComparator());
		viewer.addSelectionChangedListener(event -> removeButton.setEnabled(!event.getSelection().isEmpty()));

		TableViewerColumn column = new TableViewerColumn(viewer, SWT.NONE);
		column.getColumn().setText("Name");
		tabLayout.setColumnData(column.getColumn(), new ColumnWeightData(2));
		column.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(final Object element) {
				if (element instanceof Entry<?, ?>) {
					return ((Entry<?, ?>) element).getValue().toString();
				}
				return super.getText(element);
			}
		});
		column.setEditingSupport(new EditingSupport(viewer) {

			private final TextCellEditor editor = new TextCellEditor(viewer.getTable());

			@SuppressWarnings("unchecked")
			@Override
			protected void setValue(final Object element, final Object value) {
				if (element instanceof Entry<?, ?>) {
					if (value instanceof String) {
						final String trimmed = ((String) value).trim();
						if (!trimmed.isEmpty()) {
							((Entry<?, String>) element).setValue(trimmed);
							viewer.refresh(element);
						}
					}
				}
			}

			@Override
			protected Object getValue(final Object element) {
				if (element instanceof Entry<?, ?>) {
					return ((Entry<?, ?>) element).getValue();
				}
				return null;
			}

			@Override
			protected CellEditor getCellEditor(final Object element) {
				if (element instanceof Entry<?, ?>) {
					return editor;
				}
				return null;
			}

			@Override
			protected boolean canEdit(final Object element) {
				return element instanceof Entry<?, ?>;
			}
		});

		column = new TableViewerColumn(viewer, SWT.NONE);
		column.getColumn().setText("Standardausschnitt");
		tabLayout.setColumnData(column.getColumn(), new ColumnWeightData(1));
		column.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(final Object element) {
				if (element instanceof Entry<?, ?>) {
					if (((Entry<?, ?>) element).getKey().equals(defaultAusschnitt)) {
						return "X";
					}
					return "";
				}
				return super.getText(element);
			}
		});
		column.setEditingSupport(new EditingSupport(viewer) {

			private final CheckboxCellEditor editor = new CheckboxCellEditor(viewer.getTable());

			@Override
			protected void setValue(final Object element, final Object value) {
				if (element instanceof Entry<?, ?>) {
					if (value instanceof Boolean) {
						final Ausschnitt ausschnitt = (Ausschnitt) ((Entry<?, ?>) element).getKey();
						if ((Boolean) value) {
							defaultAusschnitt = ausschnitt;
						} else if (defaultAusschnitt == ausschnitt) {
							defaultAusschnitt = null;
						}

					}
					viewer.refresh();
				}
			}

			@Override
			protected Object getValue(final Object element) {
				if (element instanceof Entry<?, ?>) {
					return ((Entry<?, ?>) element).getKey().equals(defaultAusschnitt);
				}
				return null;
			}

			@Override
			protected CellEditor getCellEditor(final Object element) {
				if (element instanceof Entry<?, ?>) {
					return editor;
				}
				return null;
			}

			@Override
			protected boolean canEdit(final Object element) {
				return element instanceof Entry<?, ?>;
			}
		});

		removeButton = new Button(panel, SWT.PUSH);
		GridDataFactory.fillDefaults().grab(true, false).applyTo(removeButton);
		removeButton.setText("Ausschnitt löschen");
		removeButton.setEnabled(false);
		removeButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				final List<?> list = ((IStructuredSelection) viewer.getSelection()).toList();

				for (final Object obj : list) {
					final Ausschnitt ausschnit = (Ausschnitt) ((Entry<?, ?>) obj).getKey();
					if (ausschnit.equals(defaultAusschnitt)) {
						defaultAusschnitt = null;
					}
					if (obj instanceof Entry<?, ?>) {
						ausschnitte.remove(ausschnit);
					}
				}
				viewer.refresh();
			}
		});

		viewer.setInput(ausschnitte.entrySet());
		return panel;
	}

	public Map<Ausschnitt, String> getResult() {
		return ausschnitte;
	}
}
