/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.dialogs;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.Platform;
import org.eclipse.draw2d.ColorConstants;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.gef.commands.CommandStack;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTreeViewer;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.ICheckStateProvider;
import org.eclipse.jface.viewers.IColorProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerDropAdapter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DragSourceAdapter;
import org.eclipse.swt.dnd.DragSourceEvent;
import org.eclipse.swt.dnd.DropTargetEvent;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.dnd.TransferData;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;

import com.bitctrl.lib.eclipse.dialogs.ToolWindow;
import com.bitctrl.lib.eclipse.emf.gef.commands.MoveCommand;
import com.bitctrl.lib.eclipse.emf.gef.commands.SetCommand;

import de.bsvrz.buv.plugin.darstellung.model.Darstellung;
import de.bsvrz.buv.plugin.darstellung.model.DarstellungPackage;
import de.bsvrz.buv.plugin.darstellung.model.Ebene;
import de.bsvrz.buv.plugin.darstellung.util.DarstellungIcons;

/**
 * Ein schwebendes Fenster, in welchen die Ebenen ein- und ausgeblenden werden
 * können.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 * @see IAdaptable
 */
public class EbenenWindow extends ToolWindow {

	private final class EbenenViewerDropAdapter extends ViewerDropAdapter {
		private EbenenViewerDropAdapter(final Viewer viewer) {
			super(viewer);
		}

		@Override
		public boolean validateDrop(final Object target, final int operation, final TransferData transferType) {
			final IStructuredSelection isel = (IStructuredSelection) getViewer().getSelection();

			for (final Object obj : isel.toArray()) {
				if (obj instanceof Ebene && target instanceof Ebene
						&& !Objects.equals(((Ebene) obj).eContainer(), ((Ebene) target).eContainer())) {
					return false;
				}
			}
			return true;
		}

		@Override
		public boolean performDrop(final Object data) {
			final Ebene ebene = (Ebene) data;
			final DropTargetEvent event = getCurrentEvent();
			final int location = determineLocation(event);
			final Ebene target = (Ebene) determineTarget(event);

			EReference feature = DarstellungPackage.Literals.DARSTELLUNG__EBENEN;
			final EObject parent = target.eContainer();
			EList<Ebene> ebenen = getDarstellung().getEbenen();
			int targetIndex = ebenen.indexOf(target);
			if (parent instanceof Ebene) {
				ebenen = ((Ebene) parent).getEbenen();
				targetIndex = ebenen.indexOf(target);
				feature = DarstellungPackage.Literals.EBENE__EBENEN;

			}

			final MoveCommand cmd;
			switch (location) {
			case 1:
				// Dropped before the target
				if (targetIndex > 0) {
					--targetIndex;
				}

				cmd = new MoveCommand(parent, feature, ebene, targetIndex);
				break;
			case 2:
				// Dropped after the target
				if (targetIndex < ebenen.size() - 1) {
					++targetIndex;
				}

				cmd = new MoveCommand(parent, feature, ebene, targetIndex);
				break;
			case 3:
				// Dropped on the target
				cmd = new MoveCommand(parent, feature, ebene, targetIndex);
				break;
			case 4:
				// Dropped into nothing
			default:
				cmd = null;
			}
			if (cmd != null) {
				getCommandStack().execute(cmd);
				getViewer().refresh();
			}

			return true;
		}
	}

	private static final class EbenenTreeContentProvider implements ITreeContentProvider {
		@Override
		public Object[] getElements(final Object inputElement) {
			final List<?> list = new ArrayList<Object>((List<?>) inputElement);
			Collections.reverse(list);
			return list.toArray();
		}

		@Override
		public void inputChanged(final Viewer v, final Object oldInput, final Object newInput) {
			// tut nix
		}

		@Override
		public void dispose() {
			// tut nix
		}

		@Override
		public Object[] getChildren(final Object parentElement) {
			if (parentElement instanceof Ebene) {
				return ((Ebene) parentElement).getEbenen().toArray();
			}
			return new Object[0];
		}

		@Override
		public Object getParent(final Object element) {
			if (element instanceof Ebene) {
				return ((Ebene) element).eContainer();
			}
			return null;
		}

		@Override
		public boolean hasChildren(final Object element) {
			if (element instanceof Ebene) {
				return !((Ebene) element).getEbenen().isEmpty();
			}
			return false;
		}
	}

	private final class CheckStateListener implements ICheckStateListener {
		@Override
		public void checkStateChanged(final CheckStateChangedEvent event) {
			final Ebene ebene = (Ebene) event.getElement();
			if (!ebene.isAusblendbar()) {
				event.getCheckable().setChecked(ebene, true);
				final SetCommand cmd = new SetCommand(ebene, DarstellungPackage.Literals.EBENE__VISIBLE, true);
				getCommandStack().execute(cmd);
			} else {
				final SetCommand cmd = new SetCommand(ebene, DarstellungPackage.Literals.EBENE__VISIBLE,
						!ebene.isVisible());
				getCommandStack().execute(cmd);
			}
			viewer.refresh();
		}
	}

	private static final class CheckStateProvider implements ICheckStateProvider {
		@Override
		public boolean isGrayed(final Object element) {
			if (element instanceof Ebene) {
				return !((Ebene) element).isAusblendbar();
			}
			return false;
		}

		@Override
		public boolean isChecked(final Object element) {
			if (element instanceof Ebene) {
				return ((Ebene) element).isVisible();
			}
			return false;
		}
	}

	private static final class LabelProviderExtension extends LabelProvider implements IColorProvider {

		@Override
		public String getText(final Object element) {
			return ((Ebene) element).getName();
		}

		@Override
		public Color getForeground(final Object element) {
			final Ebene ebene = (Ebene) element;
			if (ebene.isVisible()) {
				return ColorConstants.black;
			}
			return ColorConstants.gray;
		}

		@Override
		public Color getBackground(final Object element) {
			return null;
		}
	}

	private CheckboxTreeViewer viewer;
	private Image ebeneVisible;
	private Image ebeneInvisible;

	{
		setShellStyle(SWT.DIALOG_TRIM | SWT.RESIZE | SWT.TOOL);
	}

	/**
	 * Initialisiert der Ebenemanager.
	 *
	 * @param part
	 *            der Workbench Part zu dem der Ebenemanager gehört.
	 * @param partControl
	 *            das Control des Workbench Parts bzw. das Control an dessen
	 *            Ecken der Ebenemanager ausgerichtet wird.
	 */
	public EbenenWindow(final IWorkbenchPart part, final Control partControl) {
		super(part, partControl);
		setCorner(Corner.TopRight);
	}

	@Override
	protected Control createContents(final Composite parent) {
		final Composite container = new Composite(parent, SWT.NONE);
		container.setLayout(new FillLayout());

		if (getDarstellung() != null) {
			createViewer(container);
		} else {
			final Label label = new Label(container, SWT.WRAP);
			label.setText("Keine Darstellung vorhanden.");
			label.setBackground(parent.getDisplay().getSystemColor(SWT.COLOR_WHITE));

		}

		return container;
	}

	private void createViewer(final Composite parent) {
		ebeneVisible = DarstellungIcons.StatusEbeneVisible.getImageDescriptor().createImage();
		ebeneInvisible = DarstellungIcons.StatusEbeneInvisible.getImageDescriptor().createImage();

		viewer = new CheckboxTreeViewer(parent, SWT.BORDER | SWT.SINGLE);
		viewer.setLabelProvider(new LabelProviderExtension());
		viewer.setContentProvider(new EbenenTreeContentProvider());

		viewer.setCheckStateProvider(new CheckStateProvider());

		viewer.setInput(getDarstellung().getEbenen());

		// Drag'n'Drop
		final Transfer[] transferTypes = { ViewerTransfer.getInstance() };
		final int operations = DND.DROP_MOVE | DND.DROP_COPY;
		viewer.addDragSupport(operations, transferTypes, new DragSourceAdapter() {

			@Override
			public void dragSetData(final DragSourceEvent event) {
				final IStructuredSelection selection = (IStructuredSelection) viewer.getSelection();
				final Object firstElement = selection.getFirstElement();

				if (ViewerTransfer.getInstance().isSupportedType(event.dataType)) {
					event.data = firstElement;
				}
			}

		});
		viewer.addDropSupport(operations, transferTypes, new EbenenViewerDropAdapter(viewer));

		viewer.addCheckStateListener(new CheckStateListener());

	}

	private Darstellung getDarstellung() {
		Darstellung result = getWorkbenchPart().getAdapter(Darstellung.class);
		if (result == null) {
			result = Platform.getAdapterManager().getAdapter(getWorkbenchPart(), Darstellung.class);
		}

		return result;
	}

	private CommandStack getCommandStack() {
		CommandStack result = getWorkbenchPart().getAdapter(CommandStack.class);
		if (result == null) {
			result = Platform.getAdapterManager().getAdapter(getWorkbenchPart(), CommandStack.class);
		}

		return result;
	}

	@Override
	protected void configureShell(final Shell newShell) {
		super.configureShell(newShell);

		newShell.setText("Ebenen");
		newShell.setSize(200, 400);
	}

	@Override
	public boolean close() {
		if (ebeneVisible != null) {
			ebeneVisible.dispose();
			ebeneVisible = null;
		}
		if (ebeneInvisible != null) {
			ebeneInvisible.dispose();
			ebeneInvisible = null;
		}

		return super.close();
	}

}
