/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.dialogs;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.Platform;
import org.eclipse.draw2d.FigureCanvas;
import org.eclipse.draw2d.LightweightSystem;
import org.eclipse.draw2d.MarginBorder;
import org.eclipse.draw2d.Viewport;
import org.eclipse.gef.GraphicalViewer;
import org.eclipse.gef.LayerConstants;
import org.eclipse.gef.RootEditPart;
import org.eclipse.gef.editparts.ScalableRootEditPart;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Canvas;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;

import com.bitctrl.lib.eclipse.dialogs.ToolWindow;
import com.bitctrl.lib.eclipse.draw2d.parts.ScrollableThumbnail2;
import com.bitctrl.lib.eclipse.draw2d.parts.Thumbnail2;

import de.bsvrz.buv.plugin.darstellung.model.Darstellung;
import de.bsvrz.buv.plugin.darstellung.model.MassstaeblicheDarstellung;

/**
 * Ein schwebendes Fenster, in welchen die Ebenen ein- und ausgeblenden werden
 * können.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 * @see IAdaptable
 */
public class OverviewWindow extends ToolWindow {

	private Canvas overview;
	private Thumbnail2 thumbnail;
	private DisposeListener disposeListener;

	/**
	 * Initialisiert das Legendenfenster.
	 *
	 * @param part
	 *            der Workbench Part zu dem das Toolfenster gehört.
	 * @param partControl
	 *            das Control des Workbench Parts bzw. das Control an dessen
	 *            Ecken das Toolfenster ausgerichtet wird.
	 */
	public OverviewWindow(final IWorkbenchPart part, final Control partControl) {
		super(part, partControl);
		setCorner(Corner.BottomRight);
	}

	@Override
	protected Control createContents(final Composite parent) {
		final Composite container = new Composite(parent, SWT.NONE);
		container.setLayout(new FillLayout());

		if (getGraphicalViewer() != null) {
			createThumbnail(container);
		} else {
			final Label label = new Label(container, SWT.WRAP);
			label.setText("Übersicht nicht verfügbar.");
			label.setBackground(parent.getDisplay().getSystemColor(SWT.COLOR_WHITE));

		}

		return container;
	}

	private Darstellung getDarstellung() {
		Darstellung result = getWorkbenchPart().getAdapter(Darstellung.class);
		if (result == null) {
			result = Platform.getAdapterManager().getAdapter(getWorkbenchPart(), Darstellung.class);
		}

		return result;
	}

	private void createThumbnail(final Composite container) {
		overview = new Canvas(container, SWT.NONE);
		final LightweightSystem lws = new LightweightSystem(overview);
		final RootEditPart rep = getGraphicalViewer().getRootEditPart();
		if (rep instanceof final ScalableRootEditPart root) {
			int maxBufferSize;
			if (getDarstellung() instanceof MassstaeblicheDarstellung) {
				maxBufferSize = 1000000;
			} else {
				maxBufferSize = 25600;
			}
			thumbnail = new ScrollableThumbnail2(maxBufferSize, (Viewport) root.getFigure());
			thumbnail.setBorder(new MarginBorder(3));
			thumbnail.setSource(root.getLayer(LayerConstants.PRINTABLE_LAYERS));
			lws.setContents(thumbnail);
			disposeListener = e -> {
				if (thumbnail != null) {
					thumbnail.deactivate();
					thumbnail = null;
				}
			};
			getEditor().addDisposeListener(disposeListener);
		}
	}

	private GraphicalViewer getGraphicalViewer() {
		GraphicalViewer result = getWorkbenchPart().getAdapter(GraphicalViewer.class);
		if (result == null) {
			result = Platform.getAdapterManager().getAdapter(getWorkbenchPart(), GraphicalViewer.class);
		}

		return result;
	}

	private FigureCanvas getEditor() {
		return (FigureCanvas) getGraphicalViewer().getControl();
	}

	@Override
	public boolean close() {
		if (disposeListener != null && getEditor() != null && !getEditor().isDisposed()) {
			getEditor().removeDisposeListener(disposeListener);
		}
		if (thumbnail != null) {
			thumbnail.deactivate();
			thumbnail = null;
		}
		return super.close();
	}

	@Override
	protected void configureShell(final Shell newShell) {
		super.configureShell(newShell);

		newShell.setText("Übersicht");
		newShell.setSize(200, 200);
		// TODO Seitenverhältnis der Darstellung auf Fenser übernehmen
		// if (getDarstellung() != null && getDarstellung().getSize() != null) {
		// final Dimension size = getDarstellung().getSize();
		// final int offset = newShell.getBorderWidth();
		// newShell.getBounds();
		// newShell.getClientArea();
		// newShell.setSize(size.width + offset, size.height + offset);
		// }
	}

}
