/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.editparts;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.emf.common.notify.Notification;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.EditPartViewer;
import org.eclipse.gef.Request;

import de.bsvrz.buv.plugin.darstellung.model.AutoEbene;
import de.bsvrz.buv.plugin.darstellung.model.DarstellungPackage;
import de.bsvrz.buv.plugin.dobj.requests.DobjRequestConstants;
import de.bsvrz.buv.plugin.dobj.requests.OnlineRequest;
import de.bsvrz.buv.plugin.dobj.tools.DObjSelectionTool;
import de.bsvrz.dav.daf.main.config.MutableSet;
import de.bsvrz.dav.daf.main.config.MutableSetChangeListener;
import de.bsvrz.dav.daf.main.config.SystemObject;

/**
 * Controller für eine Ebene der maßstäblichen Darstellung.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class AutoEbeneEditPart extends EbeneEditPart implements MutableSetChangeListener {

	private boolean online;
	private boolean pretendToBeEmpty;

	@Override
	public AutoEbene getModel() {
		return (AutoEbene) super.getModel();
	}

	@Override
	protected List<?> getModelChildren() {
		final List<Object> doModels = new ArrayList<>();
		if (!pretendToBeEmpty) {
			doModels.addAll(super.getModelChildren());
			doModels.addAll(getModel().getAutoDoObjekte());
		}
		return doModels;
	}

	/**
	 * Flag, ob die Ebene Online oder Offline geschalten ist.
	 *
	 * @return <code>true</code>, wenn die Ebene Online ist, sonst
	 *         <code>false</code>.
	 */
	public boolean isOnline() {
		return online;
	}

	/**
	 * Meldet sich auf Änderungen an den parametrierten dynamischen Mengen an.
	 */
	protected void geheOnline() {
		if (isOnline()) {
			return;
		}

		for (final MutableSet set : getModel().getMutableSets()) {
			set.addChangeListener(this);
		}
	}

	/**
	 * Meldet sich von Änderungen an den parametrierten dynamischen Mengen ab.
	 */
	protected void geheOffline() {
		if (!isOnline()) {
			return;
		}

		for (final MutableSet set : getModel().getMutableSets()) {
			set.removeChangeListener(this);
		}
	}

	/*
	 * in der deactivate-Methode muss unbedingt geheOffline ausgeführt werden,
	 * da diese Methode nicht aufgerufen wird, wenn der Editor einfach
	 * gesch,ossen wird. Falls die Ebene änderbare Objektmengen enthält, werden
	 * die entsprechenden Listener nicht abgemeldet und die Ebene wird vom GC
	 * nicht entfernt.
	 */
	@Override
	public void deactivate() {
		geheOffline();
		super.deactivate();
	}

	/**
	 * Verarbeitet ein {@link DobjRequestConstants#REQ_ONLINE}.
	 */
	@Override
	public void performRequest(final Request req) {
		super.performRequest(req);

		if (DobjRequestConstants.REQ_ONLINE.equals(req.getType())) {
			if (((OnlineRequest) req).isOnline()) {
				geheOnline();
				online = true;
			} else {
				geheOffline();
				online = false;
			}
		}
	}

	@Override
	public void notifyChanged(final Notification notification) {
		super.notifyChanged(notification);

		final Object notifier = notification.getNotifier();
		final int type = notification.getEventType();

		if (notifier instanceof AutoEbene) {
			final int featureID = notification.getFeatureID(AutoEbene.class);
			switch (type) {
			case Notification.ADD:
			case Notification.ADD_MANY:
			case Notification.MOVE:
			case Notification.REMOVE:
			case Notification.REMOVE_MANY:
				switch (featureID) {
				case DarstellungPackage.AUTO_EBENE__SYSTEM_OBJECT_TYPES:
				case DarstellungPackage.AUTO_EBENE__CONFIGURATION_AREAS:
				case DarstellungPackage.AUTO_EBENE__MUTABLE_SETS:
				case DarstellungPackage.AUTO_EBENE__SYSTEM_OBJECTS:
				case DarstellungPackage.AUTO_EBENE__SYSTEM_OBJECTS_EXCLUDES:
				case DarstellungPackage.AUTO_EBENE__DO_TYPEN:
				case DarstellungPackage.AUTO_EBENE__OBJEKT_FILTER:
					refreshChildren();
					break;
				}
				break;
			}
		}
	}

	@Override
	public void update(final MutableSet set, final SystemObject[] addedObjects, final SystemObject[] removedObjects) {
		if (getViewer().getControl() != null && !getViewer().getControl().isDisposed()) {

			getViewer().getControl().getDisplay().asyncExec(() -> {
				// TODO: add und removechildren!!!!

				// workaround:
				pretendToBeEmpty = true;
				refreshChildren();
				pretendToBeEmpty = false;
				refreshChildren();

				final EditPartViewer viewer = getViewer();
				if (viewer != null) {
					final DObjSelectionTool tool = (DObjSelectionTool) viewer
							.getProperty(DObjSelectionTool.class.getSimpleName());
					if (tool != null) {
						tool.resetChildren();
					}
				}

				if (isOnline()) {
					for (final Object child : getChildren()) {
						((EditPart) child).performRequest(new OnlineRequest(true));
					}
				}
			});
		}
	}
}
