/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.darstellung.editparts;

import java.util.List;

import org.eclipse.draw2d.ColorConstants;
import org.eclipse.draw2d.LineBorder;
import org.eclipse.draw2d.geometry.Dimension;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.util.EList;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.EditPolicy;
import org.eclipse.swt.SWT;

import de.bsvrz.buv.plugin.darstellung.editpolicies.DarstellungRootComponentEditPolicy;
import de.bsvrz.buv.plugin.darstellung.figures.DarstellungFigure;
import de.bsvrz.buv.plugin.darstellung.model.AnzeigeVerfahren;
import de.bsvrz.buv.plugin.darstellung.model.Darstellung;
import de.bsvrz.buv.plugin.darstellung.model.DarstellungPackage;
import de.bsvrz.buv.plugin.darstellung.model.Ebene;
import de.bsvrz.buv.plugin.darstellung.model.StilisierteDarstellung;
import de.bsvrz.buv.plugin.dobj.editparts.BaseGraphicalEditPart;
import de.bsvrz.buv.plugin.dobj.util.StoerfallConstants;

/**
 * Allgemeiner Controller für Darstellungen.
 *
 * @param <T>
 *            der Darstellungstyp
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public abstract class DarstellungEditPart<T extends Darstellung> extends BaseGraphicalEditPart<T, DarstellungFigure> {

	@Override
	public void setParent(final EditPart parent) {
		if (parent != null) {
			super.setParent(parent);
			getViewer().setProperty(Darstellung.class.toString(), getModel());
		} else {
			if (getParent() != null) {
				getViewer().setProperty(Darstellung.class.toString(), null);
			}
			super.setParent(null);
		}
	}

	@Override
	protected List<Ebene> getModelChildren() {
		return getModel().getEbenen();
	}

	@Override
	protected DarstellungFigure createFigure() {
		return new DarstellungFigure();
	}

	@Override
	protected void createEditPolicies() {
		installEditPolicy(EditPolicy.COMPONENT_ROLE, new DarstellungRootComponentEditPolicy());
	}

	/**
	 * Stellt den Zoom initial auf 100% und meldet Onlinedaten an, wenn die
	 * Darstellung <em>nicht</em> in einem Editor zum Bearbeiten geöffnet wurde.
	 */
	@Override
	public void activate() {
		super.activate();

		updateZoomLevels();

		// Initiale Zoomstufe auf 100% stellen
		final double zoom = 1 / getZoomManager().getUIMultiplier();
		getZoomManager().setZoom(zoom);

		if (isEditor()) {
			getFigure().setBorder(new LineBorder(ColorConstants.gray, 1, SWT.LINE_DASH));

			final EList<Ebene> ebenen = getModel().getEbenen();
			ebenen.get(ebenen.size() - 1).setEditable(true);
		}
	}

	@Override
	public void notifyChanged(final Notification notification) {
		final Object notifier = notification.getNotifier();
		final int type = notification.getEventType();

		if (notifier instanceof Darstellung) {
			final int featureID = notification.getFeatureID(Darstellung.class);
			switch (type) {
			case Notification.SET:
				switch (featureID) {
				case DarstellungPackage.DARSTELLUNG__NAME:
				case DarstellungPackage.DARSTELLUNG__SIZE:
					refreshVisuals();
					break;
				case DarstellungPackage.DARSTELLUNG__ZOOM_SKALIERUNG:
					updateZoomLevels();
					break;
				case DarstellungPackage.DARSTELLUNG__STOERFALLVERFAHREN:
					updateStoerfallverfahren();
					break;
				case DarstellungPackage.DARSTELLUNG__ANZEIGE_VERFAHREN:
					updateAnzeigeVerfahren();
					break;
				default:
					break;
				}
				break;
			case Notification.ADD:
			case Notification.ADD_MANY:
			case Notification.REMOVE:
			case Notification.REMOVE_MANY:
			case Notification.MOVE:
				switch (featureID) {
				case DarstellungPackage.DARSTELLUNG__EBENEN:
					refreshChildren();
					break;
				case DarstellungPackage.DARSTELLUNG__ZOOM_STUFEN:
					updateZoomLevels();
					break;
				default:
					break;
				}
				break;
			default:
				break;
			}
		}

		if (notifier instanceof StilisierteDarstellung) {
			final int featureID = notification.getFeatureID(StilisierteDarstellung.class);
			switch (type) {
			case Notification.ADD:
			case Notification.ADD_MANY:
			case Notification.REMOVE:
			case Notification.REMOVE_MANY:
			case Notification.MOVE:
				switch (featureID) {
				case DarstellungPackage.STILISIERTE_DARSTELLUNG__SPALTEN:
					refreshChildren();
					break;
				default:
					break;
				}
				break;
			default:
				break;
			}
		}

	}

	private void updateAnzeigeVerfahren() {
		final AnzeigeVerfahren anzeigeVerfahren = getModel().getAnzeigeVerfahren();
		getViewer().setProperty(AnzeigeVerfahren.class.getName(), anzeigeVerfahren);
	}

	private void updateStoerfallverfahren() {
		final String stoerfallverfahren = getModel().getStoerfallverfahren();
		getViewer().setProperty(StoerfallConstants.VIEWER_PROPERTY_ASPECT_STOERFALL_VERFAHREN,
				getDav().getDataModel().getAspect(stoerfallverfahren));
	}

	private void updateZoomLevels() {
		getZoomManager().setUIMultiplier(getModel().getZoomSkalierung());

		final EList<Double> zoomStufen = getModel().getZoomStufen();
		final double[] zoomLevels = new double[zoomStufen.size()];
		for (int i = 0; i < zoomLevels.length; ++i) {
			zoomLevels[i] = zoomStufen.get(i);
		}
		getZoomManager().setZoomLevels(zoomLevels);
	}

	@Override
	protected void refreshVisuals() {
		super.refreshVisuals();

		final DarstellungFigure f = getFigure();

		if (getModel().getSize() != null) {
			f.setDarstellungSize(getModel().getSize());
			f.revalidate();
		} else {
			f.setDarstellungSize(new Dimension(1000, 1000));
			f.revalidate();
		}
	}

}
