/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.editparts;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.EditPolicy;
import org.eclipse.gef.Request;
import org.eclipse.gef.editpolicies.SnapFeedbackPolicy;

import com.bitctrl.lib.eclipse.emf.util.EmfUtil;

import de.bsvrz.buv.plugin.darstellung.editpolicies.DarstellungsSpalteEditPolicy;
import de.bsvrz.buv.plugin.darstellung.editpolicies.DarstellungsSpalteXYLayoutEditPolicy;
import de.bsvrz.buv.plugin.darstellung.figures.DarstellungsSpalteFigure;
import de.bsvrz.buv.plugin.darstellung.model.DarstellungPackage;
import de.bsvrz.buv.plugin.darstellung.model.DarstellungsSpalte;
import de.bsvrz.buv.plugin.darstellung.model.Ebene;
import de.bsvrz.buv.plugin.darstellung.model.StilisierteDarstellung;
import de.bsvrz.buv.plugin.dobj.editparts.BaseGraphicalEditPart;
import de.bsvrz.buv.plugin.dobj.internal.DObjPlugin;
import de.bsvrz.buv.plugin.dobj.model.DoModel;
import de.bsvrz.buv.plugin.dobj.requests.DobjRequestConstants;
import de.bsvrz.buv.plugin.dobj.requests.OnlineRequest;
import de.bsvrz.buv.plugin.dobj.util.DobjUtil;

/**
 * Controller für eine Darstellungsspalte in der stilisierten Darstellung.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class DarstellungsSpalteEditPart extends BaseGraphicalEditPart<DarstellungsSpalte, DarstellungsSpalteFigure> {

	private final boolean debug = DObjPlugin.getDefault().isDebugging();

	private boolean online;

	@Override
	protected List<DoModel> getModelChildren() {
		final List<DoModel> result = new ArrayList<>();
		final Ebene ebene = (Ebene) getParent().getModel();
		for (final DoModel m : getModel().getDoObjekte()) {
			if (ebene == m.eContainer()) {
				result.add(m);
			}
		}
		return result;
	}

	@Override
	protected DarstellungsSpalteFigure createFigure() {
		return new DarstellungsSpalteFigure();
	}

	@Override
	public void activate() {
		super.activate();
		getModel().getDarstellung().eAdapters().add(this);
	}

	@Override
	public void deactivate() {
		((EObject) getRoot().getContents().getModel()).eAdapters().remove(this);
		super.deactivate();
	}

	@Override
	protected void createEditPolicies() {
		if (isEditor()) {
			installEditPolicy(EditPolicy.COMPONENT_ROLE, new DarstellungsSpalteEditPolicy());
			installEditPolicy(EditPolicy.LAYOUT_ROLE, new DarstellungsSpalteXYLayoutEditPolicy());
			installEditPolicy("Snap Feedback", new SnapFeedbackPolicy());
		}
	}

	@Override
	public void performRequest(final Request req) {
		super.performRequest(req);

		if (DobjRequestConstants.REQ_ONLINE.equals(req.getType())) {
			online = ((OnlineRequest) req).isOnline();

			if (debug) {
				DObjPlugin.getDefault().getLog().info("Darstellungsspalte " + EmfUtil.getText(getModel())
						+ " ist jetzt " + (online ? "online" : "offline") + ".");
			}
		}
	}

	@Override
	public void notifyChanged(final Notification notification) {
		final Object notifier = notification.getNotifier();
		final int type = notification.getEventType();

		if (notifier instanceof DarstellungsSpalte) {
			final int featureID = notification.getFeatureID(DarstellungsSpalte.class);
			switch (type) {
			case Notification.SET:
				switch (featureID) {
				case DarstellungPackage.DARSTELLUNGS_SPALTE__VISIBLE:
					for (final EditPart editPart : DobjUtil.getAllChildren(this)) {
						editPart.performRequest(new OnlineRequest(online && getModel().isVisible()));
					}
					//$FALL-THROUGH$
				case DarstellungPackage.DARSTELLUNGS_SPALTE__NAME:
				case DarstellungPackage.DARSTELLUNGS_SPALTE__WIDTH:
					refreshVisuals();
					break;
				default:
					break;
				}
				break;
			case Notification.ADD:
			case Notification.ADD_MANY:
			case Notification.REMOVE:
			case Notification.REMOVE_MANY:
			case Notification.MOVE:
				switch (featureID) {
				case DarstellungPackage.DARSTELLUNGS_SPALTE__DO_OBJEKTE:
					refreshChildren();
					break;
				default:
					break;
				}
				break;
			default:
				break;
			}
		}

		if (notifier instanceof StilisierteDarstellung) {
			final int featureID = notification.getFeatureID(StilisierteDarstellung.class);
			switch (type) {
			case Notification.SET:
				switch (featureID) {
				case DarstellungPackage.STILISIERTE_DARSTELLUNG__SPALTEN_BORDER:
					refreshVisuals();
					break;
				default:
					break;
				}
				break;
			}
		}
	}

	@Override
	protected void refreshVisuals() {
		super.refreshVisuals();

		final DarstellungsSpalteFigure f = getFigure();

		f.setVisible(getModel().isVisible());

		final StilisierteDarstellung darstellung = getModel().getDarstellung();
		if (darstellung.isSpaltenBorder()) {
			f.showBorder();
		} else {
			f.hideBorder();
		}

		final int width = getModel().getWidth();
		f.getParent().setConstraint(f, width);
	}

}
