/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.editparts.tree;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.common.notify.Notification;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.EditPolicy;
import org.eclipse.gef.RootEditPart;
import org.eclipse.gef.editpolicies.RootComponentEditPolicy;

import com.bitctrl.lib.eclipse.emf.gef.model.EListNode;

import de.bsvrz.buv.plugin.darstellung.editpolicies.DarstellungTreeEditPolicy;
import de.bsvrz.buv.plugin.darstellung.model.Darstellung;
import de.bsvrz.buv.plugin.darstellung.model.DarstellungPackage;
import de.bsvrz.buv.plugin.darstellung.model.MassstaeblicheDarstellung;
import de.bsvrz.buv.plugin.darstellung.model.StilisierteDarstellung;
import de.bsvrz.buv.plugin.dobj.editparts.tree.BaseTreeEditPart;

/**
 * Tree Edit Part für eine {@link Darstellung}.
 *
 * <p>
 * Unterstützt die {@link MassstaeblicheDarstellung} und
 * {@link StilisierteDarstellung}.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class DarstellungTreeEditPart extends BaseTreeEditPart<Darstellung> {

	private EListNode ebenen;
	private EListNode spalten;
	private EListNode doTypen;

	@Override
	public void setModel(final Object model) {
		super.setModel(model);

		ebenen = null;
		spalten = null;
		doTypen = null;
	}

	/**
	 * Gibt Zwischenknoten für die Ebenen, die Darstellungsspalten (nur
	 * stilisierte Darstellung) und die in der Darstellung verwendeten
	 * Darstellungsobjektypen zurück.
	 *
	 * <p>
	 * Der Zwischenknoten für die Darstellungsobjekttypen wird nur
	 * zurückgegeben, wenn die Darstellung bearbeitet wird. Das ist z.&nbsp;B.
	 * im Vorpositionierer der Fall.
	 */
	@Override
	protected List<?> getModelChildren() {
		if (getParent() instanceof RootEditPart) {
			return Collections.singletonList(getModel());
		}

		final List<EListNode> result = new ArrayList<>();

		if (ebenen == null) {
			ebenen = new EListNode(getModel(), DarstellungPackage.Literals.DARSTELLUNG__EBENEN);
		}
		result.add(ebenen);

		if (getModel() instanceof StilisierteDarstellung) {
			if (spalten == null) {
				spalten = new EListNode(getModel(), DarstellungPackage.Literals.STILISIERTE_DARSTELLUNG__SPALTEN);
			}
			result.add(spalten);
		}

		if (isEditor()) {
			if (doTypen == null) {
				doTypen = new EListNode(getModel(), DarstellungPackage.Literals.DARSTELLUNG__DO_TYPEN);
			}
			result.add(doTypen);
		}

		return result;
	}

	@Override
	protected void createEditPolicies() {
		super.createEditPolicies();

		if (getParent() instanceof RootEditPart) {
			installEditPolicy(EditPolicy.COMPONENT_ROLE, new RootComponentEditPolicy());
			installEditPolicy(EditPolicy.PRIMARY_DRAG_ROLE, new DarstellungTreeEditPolicy());
		}
	}

	@Override
	public void notifyChanged(final Notification notification) {
		final int type = notification.getEventType();
		final int featureID = notification.getFeatureID(Darstellung.class);

		final int size = getChildren().size();
		Map<Object, EditPart> modelToEditPart = Collections.emptyMap();
		if (size > 0) {
			modelToEditPart = new HashMap<>(size);
			for (int i = 0; i < size; i++) {
				final EditPart editPart = children.get(i);
				modelToEditPart.put(editPart.getModel(), editPart);
			}
		}

		switch (type) {
		case Notification.SET:
			switch (featureID) {
			case DarstellungPackage.DARSTELLUNG__NAME:
				refreshVisuals();
				break;
			}
		case Notification.ADD:
		case Notification.ADD_MANY:
		case Notification.REMOVE:
		case Notification.REMOVE_MANY:
		case Notification.MOVE:
			switch (featureID) {
			case DarstellungPackage.DARSTELLUNG__EBENEN:
				if (modelToEditPart.get(ebenen) != null) {
					modelToEditPart.get(ebenen).refresh();
				}
				break;
			case DarstellungPackage.STILISIERTE_DARSTELLUNG__SPALTEN:
				if (modelToEditPart.get(spalten) != null) {
					modelToEditPart.get(spalten).refresh();
				}
				break;
			case DarstellungPackage.DARSTELLUNG__DO_TYPEN:
				if (modelToEditPart.get(doTypen) != null) {
					modelToEditPart.get(doTypen).refresh();
				}
				break;
			}
			break;
		}

	}

}
