/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.editpolicies;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import org.eclipse.core.runtime.Assert;
import org.eclipse.draw2d.IFigure;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.PrecisionPoint;
import org.eclipse.draw2d.geometry.Translatable;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.GraphicalEditPart;
import org.eclipse.gef.Request;
import org.eclipse.gef.editpolicies.RootComponentEditPolicy;
import org.eclipse.gef.requests.ChangeBoundsRequest;
import org.eclipse.gef.requests.CreateRequest;

import de.bsvrz.buv.plugin.darstellung.model.Darstellung;
import de.bsvrz.buv.plugin.darstellung.model.DarstellungsSpalte;
import de.bsvrz.buv.plugin.darstellung.model.Ebene;
import de.bsvrz.buv.plugin.darstellung.model.StilisierteDarstellung;
import de.bsvrz.buv.plugin.darstellung.util.DarstellungAdapter;

public class DarstellungRootComponentEditPolicy extends RootComponentEditPolicy {

	@Override
	public EditPart getTargetEditPart(final Request request) {
		if (request instanceof ChangeBoundsRequest) {
			return getTargetEditPart((ChangeBoundsRequest) request);
		} else if (request instanceof CreateRequest) {
			return getTargetEditPart((CreateRequest) request);
		}

		return super.getTargetEditPart(request);
	}

	private EditPart getTargetEditPart(final ChangeBoundsRequest request) {
		final List<?> editParts = request.getEditParts();
		if (editParts.size() > 0) {
			final EditPart editPart = (EditPart) editParts.get(0);
			final EditPart parent = editPart.getParent();
			if (parent.getModel() instanceof DarstellungsSpalte) {
				final Point location = request.getLocation();
				final Ebene ebene = (Ebene) parent.getParent().getModel();
				final Darstellung darstellung = DarstellungAdapter.getDarstellung(ebene);
				final StilisierteDarstellung stilisierteDarstellung = (StilisierteDarstellung) darstellung;
				return getDarstellungsSpalte(location, stilisierteDarstellung);
			}
			return parent;
		}

		// fallback
		return null;
	}

	private EditPart getDarstellungsSpalte(final Point location, final StilisierteDarstellung stilisierteDarstellung) {
		int offset = 0;
		for (final DarstellungsSpalte spalte : stilisierteDarstellung.getSpalten()) {
			offset += spalte.getWidth();
			if (location.x < offset) {
				final EditPart editPart = getHost().getViewer().getEditPartRegistry().get(spalte);
				Assert.isNotNull(editPart, "editPart zu Darstellungsspalte nicht gefunden");
				return editPart;
			}
		}

		// fallback: location nicht innerhalb einer Spalte
		return null;
	}

	private EditPart getTargetEditPart(final CreateRequest request) {

		for (final Object child : getChildren(getHost())) {
			final EditPart childEditPart = (EditPart) child;
			final Ebene ebene = (Ebene) childEditPart.getModel();
			if (ebene.isEditable()) {
				final Darstellung darstellung = DarstellungAdapter.getDarstellung(ebene);
				if (darstellung instanceof final StilisierteDarstellung stilisierteDarstellung) {
					if (!stilisierteDarstellung.getSpalten().isEmpty()) {
						final Point location = getConstraintFor(request);
						return getDarstellungsSpalte(location, stilisierteDarstellung);
					}
				}

				return childEditPart;
			}
		}

		// fallback
		return null;
	}

	private static List<EditPart> getChildren(final EditPart parent) {
		final List<EditPart> result = new ArrayList<>();
		final List<EditPart> children = ((List<EditPart>) parent.getChildren()).stream()
				.filter(c -> c.getModel() instanceof Ebene).collect(Collectors.toList());
		result.addAll(children);
		result.addAll(children.stream().flatMap(c -> getChildren(c).stream()).collect(Collectors.toList()));
		return result;
	}

	private Point getConstraintFor(final CreateRequest request) {
		final Point location = new PrecisionPoint(request.getLocation());
		translateFromAbsoluteToLayoutRelative(location);
		return location;
	}

	private void translateFromAbsoluteToLayoutRelative(final Translatable t) {
		final IFigure figure = getLayoutContainer();
		figure.translateToRelative(t);
		figure.translateFromParent(t);
		final Point negatedLayoutOrigin = getLayoutOrigin().getNegated();
		t.performTranslate(negatedLayoutOrigin.x, negatedLayoutOrigin.y);
	}

	private IFigure getLayoutContainer() {
		return ((GraphicalEditPart) getHost()).getContentPane();
	}

	private Point getLayoutOrigin() {
		return getLayoutContainer().getClientArea().getLocation();
	}

}
