/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.editpolicies;

import java.util.List;

import org.eclipse.gef.EditPart;
import org.eclipse.gef.Request;
import org.eclipse.gef.RequestConstants;
import org.eclipse.gef.commands.Command;
import org.eclipse.gef.commands.CompoundCommand;
import org.eclipse.gef.commands.UnexecutableCommand;
import org.eclipse.gef.editpolicies.TreeContainerEditPolicy;
import org.eclipse.gef.requests.ChangeBoundsRequest;
import org.eclipse.gef.requests.CreateRequest;

import de.bsvrz.buv.plugin.darstellung.commands.AddEbeneToDarstellungCommand;
import de.bsvrz.buv.plugin.darstellung.model.Darstellung;
import de.bsvrz.buv.plugin.darstellung.model.Ebene;

/**
 * Tree Container Policy für eine Darstellung.
 *
 * <p>
 * Unterstützt die folgenden Requests:
 *
 * <ul>
 * <li>{@link RequestConstants#REQ_MOVE_CHILDREN}</li>
 * <li>{@link RequestConstants#REQ_MOVE}</li>
 * </ul>
 *
 * @author BitCtrl Systems GmbH, Christian Hösel
 *
 */
public class DarstellungTreeEditPolicy extends TreeContainerEditPolicy {

	@Override
	public Command getCommand(final Request req) {
		if (REQ_MOVE.equals(req.getType())) {
			return getMoveCommand((ChangeBoundsRequest) req);
		}

		return super.getCommand(req);
	}

	/**
	 * Erzeugt ein {@link #REQ_MOVE_CHILDREN} und deligiert es an den Parent
	 * Edit Part.
	 *
	 * @param req
	 *            ein {@link #REQ_MOVE} Request.
	 * @return falls möglich, ein Move-Command.
	 */
	protected Command getMoveCommand(final ChangeBoundsRequest req) {
		final EditPart parent = getHost().getParent();
		if (parent != null) {
			final ChangeBoundsRequest request = new ChangeBoundsRequest(REQ_MOVE_CHILDREN);
			request.setEditParts(getHost());
			request.setLocation(req.getLocation());
			request.setMoveDelta(req.getMoveDelta());
			return parent.getCommand(request);
		}

		return UnexecutableCommand.INSTANCE;
	}

	/**
	 * Wird nicht unterstützt.
	 */
	@Override
	protected Command getCreateCommand(final CreateRequest request) {
		return null;
	}

	@SuppressWarnings("unchecked")
	@Override
	protected Command getAddCommand(final ChangeBoundsRequest request) {

		final Darstellung darstellung = (Darstellung) getHost().getModel();
		final CompoundCommand compoundCommand = new CompoundCommand();
		for (final EditPart e : (List<EditPart>) request.getEditParts()) {
			final Object model = e.getModel();
			if (model instanceof Ebene) {
				compoundCommand.add(new AddEbeneToDarstellungCommand((Ebene) model, darstellung));
			}
		}
		if (!compoundCommand.isEmpty()) {
			return compoundCommand;
		}
		return null;
	}

	@Override
	protected Command getMoveChildrenCommand(final ChangeBoundsRequest request) {
		return UnexecutableCommand.INSTANCE;
	}

}
