/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.figures;

import java.util.HashMap;
import java.util.Map;

import org.eclipse.draw2d.AbstractLayout;
import org.eclipse.draw2d.IFigure;
import org.eclipse.draw2d.Layer;
import org.eclipse.draw2d.geometry.Dimension;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.Rectangle;

/**
 * Layout für die Spaltendarstellung. Die Spaltendarstellung dar nur
 * {@link Layer} enthalten.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class SpaltenLayout extends AbstractLayout {

	// TODO evtl. von AbstractHintLayout ableiten?

	/** Standardbreite für eine Darstellungsspalte */
	public static final int DEFAULT_SPALTENBREITE = 250;

	private final Map<IFigure, Integer> constraints = new HashMap<>();

	// TODO Höhe anhand des Inhalts automatisch berechnen
	private int height;

	{
		setObserveVisibility(true);
	}

	/**
	 * Gibt die als Constraint gesetzte Breite der Figure zurück.
	 *
	 * @param child
	 * @return die Constraint-Breite oder die Standardbreite, wenn keine
	 *         spezifische gesetzt wurde.
	 */
	protected int getWidth(final IFigure child) {
		final Integer width = (Integer) getConstraint(child);
		if (width != null) {
			return width;
		}

		return DEFAULT_SPALTENBREITE;
	}

	@Override
	public void layout(final IFigure container) {
		final Point offset = getOrigin(container);
		int x = offset.x;
		for (final Object e : container.getChildren()) {
			final IFigure child = (IFigure) e;
			if (!child.isVisible()) {
				continue;
			}

			final int width = getWidth(child);
			final Rectangle bounds = new Rectangle(x, offset.y, width, getHeight());
			x += width;
			child.setBounds(bounds);
		}
	}

	/**
	 * Legt die Höhe fest und addiert die <code>width-constraints</code> der
	 * sichtbaren Kinder.
	 */
	@Override
	protected Dimension calculatePreferredSize(final IFigure container, final int wHint, final int hHint) {
		final Dimension result = new Dimension(0, height);
		for (final Object e : container.getChildren()) {
			final IFigure child = (IFigure) e;
			if (!child.isVisible()) {
				continue;
			}

			result.width += getWidth(child);
		}

		return result;
	}

	/**
	 * Gibt die Höhe der Darstellungspalten zurück.
	 *
	 * @return die Höhe der Darstellungsspalten.
	 */
	public int getHeight() {
		return height;
	}

	/**
	 * Legt die Höhe der Darstellungsspalten fest.
	 *
	 * @param height
	 */
	public void setHeight(final int height) {
		if (this.height == height) {
			return;
		}

		this.height = height;
		invalidate();
	}

	/**
	 * Returns the origin for the given figure.
	 *
	 * @param parent
	 *            the figure whose origin is requested
	 * @return the origin
	 */
	public Point getOrigin(final IFigure parent) {
		return parent.getClientArea().getLocation();
	}

	@Override
	public Object getConstraint(final IFigure child) {
		return constraints.get(child);
	}

	/**
	 * Constraints sind {@link Integer}-Werte als gewünschte Spaltenbreite.
	 */
	@Override
	public void setConstraint(final IFigure child, final Object constraint) {
		super.setConstraint(child, constraint);
		if (constraint instanceof Integer) {
			constraints.put(child, (Integer) constraint);
		}
	}

	@Override
	public void remove(final IFigure child) {
		super.remove(child);
		constraints.remove(child);
	}

}
