/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.darstellung.search;

import java.util.Comparator;
import java.util.Map;

import org.eclipse.core.runtime.Adapters;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.draw2d.IFigure;
import org.eclipse.gef.GraphicalViewer;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.dialogs.FilteredItemsSelectionDialog;
import org.eclipse.ui.dialogs.SearchPattern;

import de.bsvrz.buv.plugin.dobj.editparts.DoModelEditPart;
import de.bsvrz.buv.plugin.dobj.internal.DObjPlugin;
import de.bsvrz.buv.plugin.dobj.model.DoModel;
import de.bsvrz.buv.rw.basislib.legende.ILegendeBaustein;

/**
 * Dialog um über alle Objekte einer Darstellung zu suchen.
 *
 * @author BitCtrl Systems GmbH, Christian
 */
public class ObjekteSucheDialog extends FilteredItemsSelectionDialog {

	/**
	 * Filter nach Filtertext.
	 *
	 */
	private final class ItemsFilterExtension extends ItemsFilter {

		ItemsFilterExtension() {
			this(new SearchPattern());
		}

		ItemsFilterExtension(final SearchPattern pattern) {
			super(pattern);
			String stringPattern = "";
			if (pattern != null && getPatternControl() instanceof final Text text) {
				stringPattern = "*" + text.getText();
			}
			patternMatcher.setPattern(stringPattern);
		}

		@Override
		public boolean matchItem(final Object item) {
			return matches(listLabelProvider.getText(item));
		}

		@Override
		public boolean isConsistentItem(final Object item) {
			return true;
		}

		@Override
		public String getPattern() {
			final String pattern = super.getPattern();
			if (pattern.isEmpty()) {
				// damit wird bei leerem Filter nichts mehr gefiltert
				return "?";
			}
			return pattern;
		}
	}

	private class SuchenLabelProvider extends LabelProvider {
		@Override
		public String getText(final Object element) {
			if (element instanceof DoModelEditPart) {
				@SuppressWarnings("unchecked")
				final DoModelEditPart<DoModel, IFigure> editpart = (DoModelEditPart<DoModel, IFigure>) element;
				final String result = !editpart.isSelectable() ? "[unsichtbar] " : "";
				return result + editpart.getSuchText();
			}
			return super.getText(element);
		}

		@Override
		public Image getImage(final Object element) {
			if (Adapters.adapt(element, ILegendeBaustein.class) instanceof final ILegendeBaustein legende) {
				return legende.getIcon();
			}
			return super.getImage(element);
		}
	}

	private final class SucheDetailsLabelProvicer extends SuchenLabelProvider {
		@Override
		public String getText(final Object element) {
			final StringBuilder result = new StringBuilder().append(super.getText(element));
			if (element instanceof DoModelEditPart) {
				@SuppressWarnings("unchecked")
				final DoModelEditPart<DoModel, IFigure> editpart = (DoModelEditPart<DoModel, IFigure>) element;
				result.append(" - ").append(editpart.getModel().getDoTyp().getName());
			}
			return result.toString();
		}

	}

	private static final String DIALOG_SETTINGS = "ObjekteSucheDialogSettings";

	private final GraphicalViewer graphicalViewer;

	private final SuchenLabelProvider listLabelProvider = new SuchenLabelProvider();

	public ObjekteSucheDialog(final Shell shell, final GraphicalViewer graphicalViewer) {
		super(shell);
		this.graphicalViewer = graphicalViewer;
		setListLabelProvider(listLabelProvider);
		setDetailsLabelProvider(new SucheDetailsLabelProvicer());
	}

	@Override
	protected Control createExtendedContentArea(final Composite parent) {
		return null;
	}

	@Override
	protected IDialogSettings getDialogSettings() {
		IDialogSettings settings = DObjPlugin.getDefault().getDialogSettings().getSection(DIALOG_SETTINGS);
		if (settings == null) {
			settings = DObjPlugin.getDefault().getDialogSettings().addNewSection(DIALOG_SETTINGS);
		}
		return settings;
	}

	@Override
	protected IStatus validateItem(final Object item) {
		return Status.OK_STATUS;
	}

	@Override
	protected ItemsFilter createFilter() {
		return new ItemsFilterExtension();
	}

	@Override
	protected Comparator<?> getItemsComparator() {
		return Comparator.comparing(Object::toString);
	}

	@Override
	protected void fillContentProvider(final AbstractContentProvider contentProvider, final ItemsFilter itemsFilter,
			final IProgressMonitor progressMonitor) throws CoreException {

		final Map<?, ?> editPartRegistry = graphicalViewer.getEditPartRegistry();
		progressMonitor.beginTask("Suche ...", editPartRegistry.size());

		editPartRegistry.values().stream().filter(DoModelEditPart.class::isInstance).map(DoModelEditPart.class::cast)
				.filter(m -> ((DoModel) m.getModel()).getDoTyp().isSuchbar()).forEach(c -> {
					contentProvider.add(c, itemsFilter);
					progressMonitor.worked(1);
				});
		progressMonitor.done();

	}

	@Override
	public String getElementName(final Object item) {
		return listLabelProvider.getText(item);
	}

}
