/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.darstellung.search;

import org.eclipse.core.commands.AbstractHandler;
import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.draw2d.FigureCanvas;
import org.eclipse.draw2d.IFigure;
import org.eclipse.draw2d.Viewport;
import org.eclipse.draw2d.geometry.Dimension;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.gef.GraphicalEditPart;
import org.eclipse.gef.GraphicalViewer;
import org.eclipse.gef.editparts.ZoomManager;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.handlers.HandlerUtil;

import de.bsvrz.buv.plugin.darstellung.model.Darstellung;

/**
 * Öffnet den Dialog zum Suchen von Objekten in Darstellungen. Das im Dialog
 * ausgewählt Objekt wird in der Darstellung selektiert und die Darstellung auf
 * das Objekt zentriert.
 *
 * @author BitCtrl Systems GmbH, Christian
 */
public class OpenDavObjektSuchenDialog extends AbstractHandler {

	@Override
	public Object execute(final ExecutionEvent event) throws ExecutionException {
		final Shell shell = HandlerUtil.getActiveShell(event);

		final IWorkbenchPart activePart = HandlerUtil.getActivePart(event);

		final GraphicalViewer graphicalViewer = activePart.getAdapter(GraphicalViewer.class);
		final Darstellung darstellung = activePart.getAdapter(Darstellung.class);
		final ZoomManager zoomManager = activePart.getAdapter(ZoomManager.class);
		final ObjekteSucheDialog d = new ObjekteSucheDialog(shell, graphicalViewer);
		if (Window.OK == d.open()) {
			final Object firstResult = d.getFirstResult();
			if (firstResult instanceof final GraphicalEditPart editPart) {
				final int zoomSuchfunktion = darstellung.getZoomSuchfunktion();
				zoomManager.setZoomAsText(zoomSuchfunktion + "%");
				selectObjekt(graphicalViewer, editPart);
			}
		}
		return null;
	}

	/**
	 * Objekt selektieren und die Darstellung auf das Objekt zentrieren.
	 *
	 * @param graphicalViewer
	 * @param editPart
	 */
	private void selectObjekt(final GraphicalViewer graphicalViewer, final GraphicalEditPart editPart) {
		final Control control = graphicalViewer.getControl();
		if (control instanceof final FigureCanvas canvas) {
			final Viewport port = canvas.getViewport();
			IFigure target = editPart.getFigure();
			final Rectangle exposeRegion = target.getBounds().getCopy();
			target = target.getParent();
			while (target != null && target != port) {
				target.translateToParent(exposeRegion);
				target = target.getParent();
			}

			final Dimension viewportSize = port.getClientArea().getSize();
			exposeRegion.expand(viewportSize.width / 2, viewportSize.height / 2);

			final Point topLeft = exposeRegion.getTopLeft();
			final Point bottomRight = exposeRegion.getBottomRight().translate(viewportSize.getNegated());
			final Point finalLocation = new Point();

			if (viewportSize.width < exposeRegion.width) {
				finalLocation.x = Math.min(bottomRight.x, Math.max(topLeft.x, port.getViewLocation().x));
			} else {
				finalLocation.x = Math.min(topLeft.x, Math.max(bottomRight.x, port.getViewLocation().x));
			}

			if (viewportSize.height < exposeRegion.height) {
				finalLocation.y = Math.min(bottomRight.y, Math.max(topLeft.y, port.getViewLocation().y));
			} else {
				finalLocation.y = Math.min(topLeft.y, Math.max(bottomRight.y, port.getViewLocation().y));
			}
			((FigureCanvas) control).scrollSmoothTo(finalLocation.x, finalLocation.y);
		} else {
			graphicalViewer.reveal(editPart);
		}

		if (editPart.isSelectable()) {
			graphicalViewer.setSelection(new StructuredSelection(editPart));
		}

	}

}
