/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.util;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.MultiStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.Diagnostic;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.util.Diagnostician;
import org.eclipse.emf.ecore.util.EcoreUtil;

import com.bitctrl.lib.eclipse.emf.dav.mock.MockDavObject;
import com.bitctrl.lib.eclipse.emf.eclipse.model.Named;

import de.bsvrz.buv.plugin.darstellung.model.Ansicht;
import de.bsvrz.buv.plugin.darstellung.model.AutoEbene;
import de.bsvrz.buv.plugin.darstellung.model.Darstellung;
import de.bsvrz.buv.plugin.dobj.internal.DObjPlugin;
import de.bsvrz.buv.plugin.dobj.model.DoModel;
import de.bsvrz.buv.plugin.dobj.model.DoTyp;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.objfilter.interpreter.Interpreter;
import de.bsvrz.sys.funclib.objfilter.interpreter.VerifizierungsFehler;

/**
 * Validator für Darstellungen und Ansichten.
 *
 * @author BitCtrl Systems GmbH, ChHoesel
 */
public final class DarstellungValidator {

	private DarstellungValidator() {
		// Konstruktor verstecken
	}

	private static IStatus createNewProblemMarker(final DoModel o) {
		final EObjectStatusImpl result = new EObjectStatusImpl(IStatus.WARNING, DObjPlugin.PLUGIN_ID, IStatus.OK,
				"Das SystemObject mit der PID " + o.getSystemObject().getPid() + " existiert nicht.", o);
		return result;
	}

	private static IStatus createNewProblemMarker(final DoTyp o) {
		final EObjectStatusImpl result = new EObjectStatusImpl(IStatus.WARNING, DObjPlugin.PLUGIN_ID, IStatus.OK,
				"Der SystemObjectTyp mit der PID " + o.getSystemObjectType().getPid() + " existiert nicht.", o);
		return result;
	}

	/**
	 * Löst alle Warnungen wegen nicht existierender {@link SystemObject}s einer
	 * Darstellung oder einer Ansicht auf.
	 *
	 * @param model
	 *            die Darstellung oder Ansicht
	 * @return {@link Status#OK_STATUS} oder einen {@link MultiStatus} mit alles
	 *         Problemen.
	 */
	public static IStatus validiereSystemObjects(final Named model) {
		final List<EObject> list = new ArrayList<>();
		model.eAllContents().forEachRemaining(list::add);

		final List<IStatus> result = findeProblematischeDarstellungsObjekte(list);
		result.addAll(findeProblematischeDarstellungsObjektTypen(list));
		result.addAll(findeProblematischeEbenen(list));
		result.addAll(findeObjektFilterProbleme(list));

		if (result.isEmpty()) {
			return Status.OK_STATUS;
		}
		return new MultiStatus(DObjPlugin.PLUGIN_ID, IStatus.WARNING, result.toArray(new IStatus[0]),
				"Die Darstellung " + model.getName() + " enthält " + result.size() + " Probleme bzw. Warnungen.", null);

	}

	/**
	 * Löst alle Probleme wegen nicht installierter Darstellungsobjekte einer
	 * Darstellung oder einer Ansicht auf.
	 *
	 * @param model
	 *            die Darstellung oder Ansicht
	 * @return {@link Status#OK_STATUS} oder einen {@link MultiStatus} mit alles
	 *         Problemen.
	 */
	public static IStatus validiereInstallierteDarstellungsobjekte(final Named model) {

		final Diagnostic diagnostic = Diagnostician.INSTANCE.validate(model);

		if (diagnostic.getSeverity() == Diagnostic.OK) {
			return Status.OK_STATUS;
		}

		final MultiStatus status = new MultiStatus(DObjPlugin.PLUGIN_ID, diagnostic.getCode(), diagnostic.getMessage(),
				diagnostic.getException());

		for (final Object name : diagnostic.getChildren()) {
			final Diagnostic childDiagnostic = (Diagnostic) name;
			final int severity = switch (childDiagnostic.getSeverity()) {
			case Diagnostic.WARNING -> IStatus.WARNING;
			case Diagnostic.ERROR -> IStatus.ERROR;
			default -> IStatus.INFO;
			};
			if (!childDiagnostic.getData().isEmpty()) {
				final MultiStatus childStatus = new MultiStatus(DObjPlugin.PLUGIN_ID, childDiagnostic.getCode(),
						childDiagnostic.getMessage(), childDiagnostic.getException());

				for (final Object obj : childDiagnostic.getData()) {

					if (obj instanceof final DoTyp typ) {
						childStatus.add(new Status(severity, DObjPlugin.PLUGIN_ID,
								"Unbekannter Darstellungsobjekttyp mit Namen \"" + typ.getName() + "\" und ID "
										+ typ.getId()
										+ " eventuell kann diese Darstellungsobjekt noch nach installiert werden.",
								null));
					}
				}

				status.add(childStatus);
			} else {
				status.add(new Status(severity, DObjPlugin.PLUGIN_ID, childDiagnostic.getMessage(),
						childDiagnostic.getException()));
			}

		}

		return status;
	}

	private static List<IStatus> findeObjektFilterProbleme(final List<EObject> list) {
		return list.parallelStream().filter(o -> o instanceof AutoEbene).map(o -> (AutoEbene) o)
				.map(e -> createObjektFilterProblemMarker(e)).filter(s -> !Status.OK_STATUS.equals(s))
				.collect(Collectors.toList());
	}

	private static IStatus createObjektFilterProblemMarker(final AutoEbene e) {

		final List<VerifizierungsFehler> fehler = e.getObjektFilter().parallelStream()
				.map(s -> Interpreter.createFilterFromString(s)).flatMap(f -> f.verifizieren().stream())
				.collect(Collectors.toList());

		if (!fehler.isEmpty()) {
			String name = e.getName();
			if (name.isEmpty()) {
				name = "[ohne Name]";
			}
			final MultiStatus childStatus = new MultiStatus(DObjPlugin.PLUGIN_ID, IStatus.WARNING,
					"Die Ebene " + name + " enthält Objektfilter mit folgenden Problemen.", null);
			fehler.stream().map(f -> new Status(IStatus.WARNING, DObjPlugin.PLUGIN_ID, f.getFehlerText()))
					.forEach(s -> childStatus.add(s));

			return childStatus;
		}
		return Status.OK_STATUS;

	}

	private static List<IStatus> findeProblematischeEbenen(final List<EObject> list) {

		return list.stream().filter(o -> o instanceof AutoEbene).map(o -> (AutoEbene) o).filter(e -> hatEbeneFehler(e))
				.map(e -> createNewEbneneProblemMarker(e)).collect(Collectors.toList());
	}

	private static IStatus createNewEbneneProblemMarker(final AutoEbene e) {
		final List<SystemObject> mocks = e.getSystemObjects().stream().filter(o -> o instanceof MockDavObject)
				.collect(Collectors.toList());
		mocks.addAll(e.getSystemObjectsExcludes().stream().filter(o -> o instanceof MockDavObject)
				.collect(Collectors.toList()));
		mocks.addAll(
				e.getSystemObjectTypes().stream().filter(o -> o instanceof MockDavObject).collect(Collectors.toList()));
		mocks.addAll(e.getConfigurationAreas().stream().filter(o -> o instanceof MockDavObject)
				.collect(Collectors.toList()));

		return new EObjectStatusImpl(IStatus.WARNING, DObjPlugin.PLUGIN_ID, IStatus.OK,
				"Die Autoebene " + e.getName() + " enthält " + mocks.size() + " fehlerhafte Referenzen ("
						+ mocks.stream().map(SystemObject::getPid).collect(Collectors.joining(",")) + ")",
				e);

	}

	private static boolean hatEbeneFehler(final AutoEbene e) {
		boolean result = e.getSystemObjects().stream().anyMatch(o -> o instanceof MockDavObject);
		result |= e.getSystemObjectsExcludes().stream().anyMatch(o -> o instanceof MockDavObject);
		result |= e.getSystemObjectTypes().stream().anyMatch(o -> o instanceof MockDavObject);
		result |= e.getConfigurationAreas().stream().anyMatch(o -> o instanceof MockDavObject);
		return result;
	}

	private static List<IStatus> findeProblematischeDarstellungsObjektTypen(final List<EObject> list) {
		return list.stream().filter(o -> o instanceof DoTyp).map(o -> (DoTyp) o)
				.filter(o -> o.getSystemObjectType() instanceof MockDavObject).map(o -> createNewProblemMarker(o))
				.collect(Collectors.toList());
	}

	private static List<IStatus> findeProblematischeDarstellungsObjekte(final List<EObject> list) {
		return list.stream().filter(o -> o instanceof DoModel).map(o -> (DoModel) o)
				.filter(o -> o.getSystemObject() instanceof MockDavObject).map(o -> createNewProblemMarker(o))
				.collect(Collectors.toList());
	}

	/**
	 * Löst alle Probleme und Warnungen einer Darstellung oder einer Ansicht
	 * auf.
	 *
	 * @param model
	 *            die Darstellung oder Ansicht
	 * @return {@link Status#OK_STATUS} oder einen {@link MultiStatus} mit alles
	 *         Problemen.
	 */
	public static List<IStatus> validiere(final Named model) {
		final IStatus validiereParserFehler = validiereParserFehler(model);
		final IStatus validiereInstallierteDarstellungsobjekte = validiereInstallierteDarstellungsobjekte(model);
		final IStatus validiereSystemObjects = validiereSystemObjects(model);
		if (Status.OK_STATUS.equals(validiereParserFehler) && Status.OK_STATUS.equals(validiereSystemObjects)
				&& Status.OK_STATUS.equals(validiereInstallierteDarstellungsobjekte)) {
			return Arrays.asList(Status.OK_STATUS);
		}

		final List<IStatus> result = new ArrayList<>();
		if (!Status.OK_STATUS.equals(validiereParserFehler)) {
			result.add(validiereParserFehler);
		}
		if (!Status.OK_STATUS.equals(validiereInstallierteDarstellungsobjekte)) {
			result.add(validiereInstallierteDarstellungsobjekte);
		}
		if (!Status.OK_STATUS.equals(validiereSystemObjects)) {
			result.add(validiereSystemObjects);
		}
		return result;
	}

	private static IStatus validiereParserFehler(final Named model) {

		EMFIDModellEinstellungen<?> einstellungen = null;

		if (model instanceof Ansicht) {
			einstellungen = AnsichtenEinstellungen.INSTANCE;
		} else if (model instanceof Darstellung) {
			einstellungen = DarstellungenEinstellungen.INSTANCE;
		} else {
			return Status.OK_STATUS;
		}

		final List<org.eclipse.emf.ecore.resource.Resource.Diagnostic> modellEinstellungDiagnostic = einstellungen
				.getModellEinstellungDiagnostic(EcoreUtil.getID(model));

		final List<Status> list = modellEinstellungDiagnostic.stream()
				.map(w -> new Status(IStatus.ERROR, DObjPlugin.PLUGIN_ID, w.getMessage())).collect(Collectors.toList());

		if (!list.isEmpty()) {
			final MultiStatus result = new MultiStatus(DObjPlugin.PLUGIN_ID, IStatus.ERROR,
					"Die Darstellung hat " + list.size() + " Parser Fehler", null);
			list.stream().forEach(s -> result.add(s));
			return result;
		}

		return Status.OK_STATUS;
	}

}