/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.util;

import java.io.IOException;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource.Diagnostic;
import org.eclipse.emf.ecore.util.EcoreUtil;

import de.bsvrz.buv.plugin.dobj.internal.DObjPlugin;
import de.bsvrz.buv.plugin.dobj.internal.RahmenwerkService;
import de.bsvrz.buv.rw.basislib.einstellungen.Einstellungen;
import de.bsvrz.buv.rw.basislib.einstellungen.EinstellungsAdresse;
import de.bsvrz.buv.rw.basislib.einstellungen.SpeicherKey;
import de.bsvrz.buv.rw.bitctrl.eclipse.modell.emf.EMFModellEinstellungen;
import de.bsvrz.buv.rw.bitctrl.eclipse.modell.emf.storage.Named;
import de.bsvrz.puk.param.lib.daten.UrlasserInfo;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Laden und Speichern von EMF-Objekten, die ein ID Attribute haben und jeweils
 * getrennt voneinander gespeichert werden.
 *
 * @author BitCtrl Systems GmbH, ChHoesel
 * @param <T>
 */
public class EMFIDModellEinstellungen<T extends EObject> extends EMFModellEinstellungen<T> {

	private static final Debug logger = Debug.getLogger();

	private static final long serialVersionUID = 1L;

	public EMFIDModellEinstellungen(final Class<T> type, final EClass eClass) {
		super(type, eClass);
	}

	@Override
	public void setModellEinstellungen(final SpeicherKey art, final String name, final T object) throws IOException {
		setModellEinstellungen(art, name, object, null);
	}

	@Override
	public void setModellEinstellungen(final SpeicherKey art, final String name, final T object,
			final UrlasserInfo urlasser) throws IOException {
		final Einstellungen verwaltung = RahmenwerkService.getService().getEinstellungen();

		// implizietes Löschen und Löschen aller Einstellungen mit gleichem
		// Namen am gleichen Speicherort (auch mit anderer ID)
		super.setModellEinstellungen(art, name, null, urlasser);

		// implizietes Löschen
		delete(art, name, urlasser);

		if (object != null) {

			final EinstellungsAdresse adresse = new EinstellungsAdresse(null,
					getTypeClass().getName() + ":" + EcoreUtil.getID(object), art.getOwnerType(), art.getPid(),
					art.getLocation());
			String param = (String) verwaltung.getValue(adresse);
			final List<Named> contents = getModellEinstellungenInternal(param);
			if (null != contents) {
				boolean hasSet = false;
				for (final Named named : contents) {
					final String contentsName = getName(named);
					if (contentsName != null && contentsName.equals(name)) {
						setObject(named, object);
						setLastModification(named, new Date());
						hasSet = true;
						break;
					}
				}
				if (!hasSet) {
					final Named named = createStorage();
					setName(named, name);
					setObject(named, object);
					setLastModification(named, new Date());
					contents.add(named);
				}
			}

			param = saveModellEinstellungenInternal(contents);
			if (urlasser != null) {
				verwaltung.setValue(adresse, param, urlasser);
			} else {
				verwaltung.setValue(adresse, param);
			}
			firePropertyChanged(PROP_EINSTELLUNG_CHANGED, null, name);
		}

	}

	/**
	 * Löschen eines gespeichert EMF Objektes.
	 *
	 */
	private void delete(final SpeicherKey art, final String name, final UrlasserInfo urlasser) throws IOException {
		final Einstellungen verwaltung = RahmenwerkService.getService().getEinstellungen();
		final Map<String, String> einstellungsIds = verwaltung.getEinstellungsId(art);

		final List<String> ansichtsKeys = einstellungsIds.entrySet().stream()
				.filter(e -> e.getKey().startsWith(getTypeClass().getName())).map(e -> e.getKey())
				.collect(Collectors.toList());
		for (final String key : ansichtsKeys) {
			final EinstellungsAdresse adresse = new EinstellungsAdresse(null, key, art.getOwnerType(), art.getPid(),
					art.getLocation());
			String param = (String) verwaltung.getValue(adresse);
			final List<Named> contents = getModellEinstellungenInternal(param);
			if (null != contents) {

				final Optional<Named> optional = contents.stream().filter(c -> Objects.equals(getName(c), name))
						.findFirst();
				if (optional.isPresent()) {
					contents.remove(optional.get());
					param = saveModellEinstellungenInternal(contents);
					verwaltung.setValue(adresse, param, urlasser);
					firePropertyChanged(PROP_EINSTELLUNG_CHANGED, null, name);
				}
				if (contents.isEmpty()) {
					verwaltung.removeValue(adresse, urlasser);
				}
			}
		}
	}

	@Override
	protected List<Named> getModellEinstellungenList(final SpeicherKey art) {

		final Einstellungen einstellungen = RahmenwerkService.getService().getEinstellungen();

		try {
			final Map<String, String> einstellungsIds = einstellungen.getEinstellungsId(art);

			final List<String> ansichtsKeys = einstellungsIds.entrySet().stream()
					.filter(e -> e.getKey().startsWith(getTypeClass().getName())).map(e -> e.getKey())
					.collect(Collectors.toList());

			return ansichtsKeys.stream().map(s -> {
				try {
					return (String) einstellungen.getValue(
							new EinstellungsAdresse(null, s, art.getOwnerType(), art.getPid(), art.getLocation()));
				} catch (final IOException ex) {
					DObjPlugin.getDefault().getLog().log(new Status(IStatus.WARNING, DObjPlugin.PLUGIN_ID,
							"Lesen der Benutzereinstellungen " + art + " - " + s + " ist fehlgeschlagen.", ex));
				}
				return null;
			}).filter(p -> p != null).flatMap(p -> getModellEinstellungenInternal(p).stream())
					.collect(Collectors.toList());
		} catch (final IOException ex) {
			logger.warning(ex.getLocalizedMessage());
		}
		return Collections.emptyList();
	}

	/**
	 * Liefert das EMF Objekt, mit der entsprechenden ID aus dem
	 * Einstellungspeicher.
	 *
	 */
	@SuppressWarnings("unchecked")
	public T getModellEinstellungbyId(final String id) {
		final Einstellungen einstellungen = RahmenwerkService.getService().getEinstellungen();

		for (final SpeicherKey art : SpeicherKey.getDefaultKeys()) {
			try {
				final Map<String, String> einstellungsIds = einstellungen.getEinstellungsId(art);

				final List<String> ansichtsKeys = einstellungsIds.entrySet().stream()
						.filter(e -> Objects.equals(e.getKey(), getTypeClass().getName() + ":" + id))
						.map(e -> e.getKey()).collect(Collectors.toList());

				return ansichtsKeys.stream().map(s -> {
					try {
						return (String) einstellungen.getValue(
								new EinstellungsAdresse(null, s, art.getOwnerType(), art.getPid(), art.getLocation()));
					} catch (final IOException ex) {
						DObjPlugin.getDefault().getLog().log(new Status(IStatus.WARNING, DObjPlugin.PLUGIN_ID,
								"Lesen der Benutzereinstellungen " + art + " - " + s + " ist fehlgeschlagen.", ex));
					}
					return null;
				}).filter(p -> p != null).flatMap(p -> getModellEinstellungenInternal(p).stream())
						.map(n -> (T) n.getObject()).findFirst().orElse(null);
			} catch (final IOException ex) {
				logger.warning(ex.getLocalizedMessage());
			}
		}
		return null;
	}

	public List<Diagnostic> getModellEinstellungDiagnostic(final String id) {
		if (id == null || id.isEmpty()) {
			return Collections.emptyList();
		}
		final Einstellungen einstellungen = RahmenwerkService.getService().getEinstellungen();

		for (final SpeicherKey art : SpeicherKey.getDefaultKeys()) {
			try {
				final Map<String, String> einstellungsIds = einstellungen.getEinstellungsId(art);

				final List<String> ansichtsKeys = einstellungsIds.entrySet().stream()
						.filter(e -> Objects.equals(e.getKey(), getTypeClass().getName() + ":" + id))
						.map(e -> e.getKey()).collect(Collectors.toList());

				final Optional<String> parameter = ansichtsKeys.stream().map(s -> {
					try {
						return (String) einstellungen.getValue(
								new EinstellungsAdresse(null, s, art.getOwnerType(), art.getPid(), art.getLocation()));
					} catch (final IOException ex) {
						DObjPlugin.getDefault().getLog().log(new Status(IStatus.WARNING, DObjPlugin.PLUGIN_ID,
								"Lesen der Benutzereinstellungen " + art + " - " + s + " ist fehlgeschlagen.", ex));
					}
					return null;
				}).filter(p -> p != null).findFirst();

				if (parameter.isPresent()) {
					return getModellEinstellungenDiagnostic(parameter.get());
				}

			} catch (final IOException ex) {
				logger.warning(ex.getLocalizedMessage());
			}
		}

		return Collections.emptyList();
	}

}