/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.views.ansicht;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Collections;
import java.util.Comparator;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;

import org.eclipse.core.runtime.Platform;
import org.eclipse.draw2d.ColorConstants;
import org.eclipse.draw2d.ToolTipHelper;
import org.eclipse.gef.EditDomain;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.EditPartViewer;
import org.eclipse.gef.GraphicalViewer;
import org.eclipse.gef.editparts.ScalableRootEditPart;
import org.eclipse.gef.editparts.ZoomManager;
import org.eclipse.gef.ui.actions.ActionRegistry;
import org.eclipse.gef.ui.parts.DomainEventDispatcher;
import org.eclipse.gef.ui.parts.ScrollingGraphicalViewer;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;

import de.bsvrz.buv.plugin.darstellung.editparts.DarstellungEditPartFactory;
import de.bsvrz.buv.plugin.darstellung.model.Ansicht;
import de.bsvrz.buv.plugin.darstellung.model.Ebene;
import de.bsvrz.buv.plugin.dobj.internal.DObjPlugin;
import de.bsvrz.buv.plugin.dobj.internal.RahmenwerkService;
import de.bsvrz.buv.plugin.dobj.model.DoModel;
import de.bsvrz.buv.plugin.dobj.model.DoTyp;
import de.bsvrz.buv.plugin.dobj.preferences.DobjPreference;
import de.bsvrz.buv.plugin.dobj.requests.OnlineRequest;
import de.bsvrz.buv.plugin.dobj.util.DobjUtil;
import de.bsvrz.buv.rw.basislib.legende.ILegendeBaustein;

/**
 * @author BitCtrl Systems GmbH, peuker
 * @author BitCtrl Systems GmbH, Thomas Thierfelder
 */
public class AnsichtComposite extends Composite implements PropertyChangeListener, DisposeListener {

	public class AnsichtEventDispatcher extends DomainEventDispatcher {

		public AnsichtEventDispatcher(final EditDomain d, final EditPartViewer v) {
			super(d, v);
		}

		@Override
		protected ToolTipHelper getToolTipHelper() {
			final ToolTipHelper helper = super.getToolTipHelper();
			helper.setHideDelay(getTooltipHideDelay());
			return helper;
		}

		private int getTooltipHideDelay() {
			String tooltipTimeoutStr = DObjPlugin.getDefault().getPreferenceStore()
					.getString(DobjPreference.TOOLTIP_HIDEDELAY.getId());
			if (tooltipTimeoutStr == null || tooltipTimeoutStr.trim().isEmpty()) {
				tooltipTimeoutStr = RahmenwerkService.getService().getRahmenWerk().getRahmenwerkProductProperties()
						.getDefaultDobjTooltipHideDelay();
			}

			if (tooltipTimeoutStr != null && !tooltipTimeoutStr.trim().isEmpty()) {
				try {
					final int result = Integer.parseInt(tooltipTimeoutStr.trim());
					return result;
				} catch (final NumberFormatException ex) {
					// ignore exception and return default value
				}
			}

			return 5000;
		}

	}

	public class AnsichtGraphicalViewer extends ScrollingGraphicalViewer {

		@Override
		public void setEditDomain(final EditDomain domain) {
			super.setEditDomain(domain);
			getLightweightSystem().setEventDispatcher(new AnsichtEventDispatcher(domain, this));

		}
	}

	private final EditDomain editDomain = new EditDomain();
	private final ActionRegistry actionRegistry = new ActionRegistry();
	private final ScalableRootEditPart rootEditPart = new ScalableRootEditPart();
	private final GraphicalViewer graphicalViewer;
	private final Ansicht ansicht;
	private boolean online;

	public AnsichtComposite(final Composite parent, final int style, final Ansicht ansicht) {
		super(parent, style);
		this.ansicht = ansicht;
		graphicalViewer = createGraphicalViewer(this);

		setLayout(new FillLayout());
		configureGraphicalViewer(graphicalViewer);
		editDomain.addViewer(graphicalViewer);

		RahmenwerkService.getService().getObjektFactory().addPropertyChangeListener(this);

		addDisposeListener(this);
	}

	private GraphicalViewer createGraphicalViewer(final Composite parent) {
		final GraphicalViewer result = new AnsichtGraphicalViewer();
		result.createControl(parent);
		result.getControl().setBackground(ColorConstants.listBackground);
		return result;
	}

	private void configureGraphicalViewer(final GraphicalViewer viewer) {
		viewer.setRootEditPart(rootEditPart);
		viewer.setEditPartFactory(new DarstellungEditPartFactory());
		viewer.setContents(ansicht);
		geheOnline();

	}

	private void geheOnline() {
		if (!online) {
			Display.getDefault().asyncExec(() -> {
				if (RahmenwerkService.getService().getObjektFactory().isVerbunden()) {
					for (final EditPart editPart : DobjUtil.getAllChildren(rootEditPart)) {
						editPart.performRequest(new OnlineRequest(true));
					}
					online = true;
				}
			});
		}
	}

	private void geheOffline() {
		if (online) {
			for (final EditPart editPart : DobjUtil.getAllChildren(rootEditPart)) {
				editPart.performRequest(new OnlineRequest(false));
			}
			online = false;
		}
	}

	@Override
	public void widgetDisposed(final DisposeEvent e) {
		RahmenwerkService.getService().getObjektFactory().removePropertyChangeListener(this);
		editDomain.setActiveTool(null);
		actionRegistry.dispose();
		graphicalViewer.setContents(null);
	}

	public ActionRegistry getActionRegistry() {
		return actionRegistry;
	}

	public ZoomManager getZoomManager() {
		return rootEditPart.getZoomManager();
	}

	public EditDomain getEditDomain() {
		return editDomain;
	}

	@Override
	public void propertyChange(final PropertyChangeEvent evt) {
		if (RahmenwerkService.getService().getObjektFactory().isVerbunden()) {
			geheOnline();
		} else {
			geheOffline();
		}
	}

	/**
	 * Liefert alle {@link ILegendeBaustein}e für diese Ansicht.
	 */
	public Map<DoTyp, ILegendeBaustein> getBausteine() {
		return getBausteine(rootEditPart);
	}

	private Map<DoTyp, ILegendeBaustein> getBausteine(final EditPart editPart) {
		final SortedMap<DoTyp, ILegendeBaustein> bausteine = new TreeMap<>(Comparator.comparing(DoTyp::getName));
		ILegendeBaustein baustein = editPart.getAdapter(ILegendeBaustein.class);
		if (baustein == null) {
			baustein = Platform.getAdapterManager().getAdapter(editPart, ILegendeBaustein.class);
		}

		// Nur sichtbare Ebenen berücksichtigen
		if (editPart.getModel() instanceof Ebene) {
			final Ebene ebene = (Ebene) editPart.getModel();
			if (!ebene.isVisible()) {
				return Collections.emptyMap();
			}
		}

		final Object model = editPart.getModel();
		if (model instanceof DoModel) {
			final DoTyp doTyp = ((DoModel) model).getDoTyp();
			if (baustein != null && !bausteine.containsKey(doTyp)) {
				bausteine.put(doTyp, baustein);
			}
		}

		for (final Object e : editPart.getChildren()) {
			bausteine.putAll(getBausteine((EditPart) e));
		}
		return bausteine;
	}

	public final GraphicalViewer getGraphicalViewer() {
		return graphicalViewer;
	}
}
