/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.wizards;

import java.io.File;
import java.io.IOException;

import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.ui.IExportWizard;
import org.eclipse.ui.PlatformUI;

import com.bitctrl.lib.eclipse.emf.eclipse.model.Named;
import com.bitctrl.lib.eclipse.emf.util.EmfUtil;

/**
 * Basisklasse für einen Wizard, die ein in den Rahmenwerkseinstellungen
 * gespeichertes {@link Named}-Objekt exportiert.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public abstract class AbstractExportWizard extends AbstractWorkbenchWizard implements IExportWizard {

	private AbstractExportWizardPage exportPage;

	/**
	 * Initialisiert den Wizard.
	 *
	 * @param eClass
	 *            die Klasse der zu exportierenden Objekte.
	 */
	protected AbstractExportWizard(final EClass eClass) {
		super(eClass);
	}

	/**
	 * Legt nur eine Wizard Page für die Exporteinstellungen an.
	 *
	 * @see #createExportPage()
	 */
	@Override
	public void addPages() {
		exportPage = createExportPage();
		exportPage.setPreSelection(getSelection());
		addPage(exportPage);
	}

	/**
	 * Erzeugt eine neue Wizard Page für die Exporteinstellungen.
	 */
	protected abstract AbstractExportWizardPage createExportPage();

	@Override
	public boolean performFinish() {
		final String directory = exportPage.getDirectory();

		final ResourceSet resSet = new ResourceSetImpl();
		final IStructuredSelection export = (IStructuredSelection) exportPage.getSelection();
		for (final Object e : export.toList()) {
			if (!getEClass().isInstance(e)) {
				continue;
			}

			final Named object = (Named) e;
			final File file = new File(directory + "/" + object.getName() + "." + getEClass().getName().toLowerCase());

			// Prüfen ob die zu schreibende Datei bereits existiert
			if (file.exists() && !MessageDialog.openQuestion(getShell(), "Datei überschreiben",
					"Es existiert bereits eine Datei mit dem selben Namen " + file
							+ ". Wollen Sie diese überschreiben?")) {
				return false;
			}

			// Datei schreiben
			final Resource resource = resSet.createResource(URI.createURI(file.toURI().toString()));
			resource.getContents().add(object);
			try {
				resource.save(null);
			} catch (final IOException ex) {
				final String message = "Die Datei konnte nicht geschrieben werden: " + EmfUtil.getText(object);
				ErrorDialog.openError(PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell(),
						"Fehler beim Export", message, Status.error(message, ex));

				return false;
			}
		}

		return true;
	}

}
