/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.wizards;

import org.eclipse.emf.ecore.EClass;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.DirectoryDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import com.bitctrl.lib.eclipse.emf.viewers.EmfLabelProvider;

import de.bsvrz.buv.rw.bitctrl.eclipse.modell.emf.EMFModellEinstellungen;
import de.bsvrz.buv.rw.bitctrl.eclipse.viewers.EinstellungenContentProvider;

/**
 * Basisklasse für eine Wizard Page, die die notwendigen Informationen (Objekte
 * und Verzeichnis) für den Export bestimmt.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public abstract class AbstractExportWizardPage extends WizardPage {

	private final EClass eClass;
	private final EMFModellEinstellungen<?> einstellungen;

	private TreeViewer viewer;
	private Text directoryText;
	private ISelection preSelection;

	/**
	 * Initialisiert die Page.
	 */
	protected AbstractExportWizardPage(final String pageName, final EClass eClass,
			final EMFModellEinstellungen<?> einstellungen) {
		super(pageName);

		this.eClass = eClass;
		this.einstellungen = einstellungen;

		setPageComplete(false);
	}

	@Override
	public void createControl(final Composite parent) {
		final Composite container = new Composite(parent, SWT.NONE);
		container.setLayout(new GridLayout(3, false));

		createViewer(container);
		createDirectory(container);

		viewer.getControl().setFocus();
		setControl(container);
	}

	private void createViewer(final Composite container) {
		final Label label = new Label(container, SWT.NONE);
		label.setText(eClass.getName());
		label.setLayoutData(GridDataFactory.swtDefaults().align(SWT.LEFT, SWT.TOP).create());

		viewer = new TreeViewer(container, SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL | SWT.BORDER);
		viewer.getControl().setLayoutData(GridDataFactory.fillDefaults().grab(true, true).span(2, 1).create());
		viewer.setContentProvider(new EinstellungenContentProvider());
		viewer.setLabelProvider(new EmfLabelProvider());
		viewer.setInput(einstellungen);
		viewer.setSelection(getPreSelection(), true);
		viewer.addSelectionChangedListener(event -> updatePageComplete());
	}

	private void createDirectory(final Composite container) {
		Label label;
		label = new Label(container, SWT.NONE);
		label.setText("Zielverzeichnis");

		directoryText = new Text(container, SWT.BORDER);
		directoryText
				.setLayoutData(GridDataFactory.swtDefaults().align(SWT.FILL, SWT.CENTER).grab(true, false).create());
		directoryText.addModifyListener(e -> updatePageComplete());

		final Button fileChooseButton = new Button(container, SWT.PUSH);
		fileChooseButton.setText("Auswählen");
		fileChooseButton.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(final SelectionEvent e) {
				final DirectoryDialog dlg = new DirectoryDialog(getShell());
				dlg.setText(getTitle());
				dlg.setMessage(getDescription());
				final String selected = dlg.open();
				if (selected != null) {
					directoryText.setText(selected);
					updatePageComplete();
				}
			}

		});
	}

	private void updatePageComplete() {
		if (viewer.getSelection().isEmpty()) {
			setErrorMessage("Sie müssen mindestens ein zu exportierendes Objekt auswählen.");
			setPageComplete(false);
		} else if (directoryText.getText().isEmpty()) {
			setErrorMessage(
					"Sie müssen das Verzeichnis angeben, in das die ausgewählten Objekte exportiert werden sollen.");
			setPageComplete(false);
		} else {
			setErrorMessage(null);
			setPageComplete(true);
		}
	}

	/**
	 * Gibt das Zielverzeichnis für den Export zurück.
	 */
	public String getDirectory() {
		return directoryText.getText();
	}

	/**
	 * Gibt die zu exportierten Objekte zurück.
	 */
	public ISelection getSelection() {
		return viewer.getSelection();
	}

	/**
	 * Gibt die Vorauswahl für die zu exportierenden Objekte zurück.
	 */
	public ISelection getPreSelection() {
		return preSelection;
	}

	/**
	 * Legt eine Vorauswahl für die zu exportierenden Objekte fest.
	 */
	public void setPreSelection(final ISelection preSelection) {
		this.preSelection = preSelection;
	}

}
