/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.wizards;

import java.io.File;
import java.io.IOException;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.MultiStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.Resource.Diagnostic;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.PlatformUI;

import com.bitctrl.lib.eclipse.emf.eclipse.model.Named;

import de.bsvrz.buv.plugin.dobj.internal.DObjPlugin;
import de.bsvrz.buv.rw.basislib.einstellungen.EinstellungLocation;
import de.bsvrz.buv.rw.basislib.einstellungen.EinstellungOwnerType;
import de.bsvrz.buv.rw.basislib.einstellungen.SpeicherKey;
import de.bsvrz.buv.rw.basislib.einstellungen.UrlasserDialogAbgebrochenException;
import de.bsvrz.buv.rw.basislib.urlasser.UrlasserInfoDatenDialog;
import de.bsvrz.buv.rw.bitctrl.eclipse.modell.emf.EMFModellEinstellungen;
import de.bsvrz.buv.rw.bitctrl.eclipse.wizards.EinstellungsArtWizardPage;
import de.bsvrz.puk.param.lib.daten.UrlasserInfo;

/**
 * Basisklasse für Wizards die {@link EObject}-Objekte importieren, die in den
 * Rahmenwerkseinstellungen gespeichert werden.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public abstract class AbstractImportWizard extends AbstractWorkbenchWizard {

	private static final class QuestionErrorDialog extends ErrorDialog {

		public QuestionErrorDialog(final Shell parentShell, final String dialogTitle, final String message,
				final IStatus status, final int displayMask) {
			super(parentShell, dialogTitle, message, status, displayMask);
		}

		@Override
		protected void createButtonsForButtonBar(final Composite parent) {
			// create OK and Details buttons
			createButton(parent, IDialogConstants.YES_ID, IDialogConstants.YES_LABEL, true);
			createButton(parent, IDialogConstants.NO_ID, IDialogConstants.NO_LABEL, false);
			createDetailsButton(parent);
		}

		@Override
		protected void buttonPressed(final int buttonId) {
			switch (buttonId) {
			case IDialogConstants.YES_ID -> {
				setReturnCode(OK);
				close();
			}
			case IDialogConstants.NO_ID -> {
				setReturnCode(CANCEL);
				close();
			}
			default -> {
				super.buttonPressed(buttonId);
			}
			}
		}
	}

	private final EMFModellEinstellungen einstellungen;

	private AbstractImportWizardPage importPage;
	private EinstellungsArtWizardPage einstellungsArtWizardPage;

	/**
	 * Initialisiert den Wizard.
	 *
	 * @param eClass
	 *            der Typ der zu importierenden Objekte.
	 */
	protected AbstractImportWizard(final EClass eClass, final EMFModellEinstellungen<?> einstellungen) {
		super(eClass);
		this.einstellungen = einstellungen;
	}

	@Override
	public void addPages() {
		importPage = getImportWizardPage();
		addPage(importPage);

		einstellungsArtWizardPage = new EinstellungsArtWizardPage(EinstellungsArtWizardPage.class.getName());
		final Set<SpeicherKey> einstellungsArten = new HashSet<>();
		for (final Object e : getSelection().toList()) {
			if (e instanceof final SpeicherKey key) {
				einstellungsArten.add(key);
			}
		}
		einstellungsArtWizardPage.setPreSelection(einstellungsArten);
		addPage(einstellungsArtWizardPage);
	}

	protected abstract AbstractImportWizardPage getImportWizardPage();

	@Override
	public boolean performFinish() {
		final List<String> fileNames = importPage.getFileNames();
		for (final SpeicherKey einstellungsArt : einstellungsArtWizardPage.getEinstellungsArten()) {

			if (einstellungsArt.getLocation() == EinstellungLocation.NETZWERKWEIT
					&& einstellungsArt.getOwnerType() == EinstellungOwnerType.SYSTEM) {

				final UrlasserInfoDatenDialog dialog = new UrlasserInfoDatenDialog(getShell(),
						(verbindung, urlasser) -> {
							for (final String fileName : fileNames) {
								final IStatus status = dateiImportieren(einstellungsArt, fileName, urlasser);
								if (status.matches(IStatus.ERROR)) {
									return;
								}
							}
						});
				dialog.open();
			} else {
				for (final String fileName : fileNames) {
					final IStatus status = dateiImportieren(einstellungsArt, fileName, null);
					if (status.matches(IStatus.ERROR)) {
						return false;
					}
				}

			}
		}

		return true;
	}

	/**
	 * Einlesen einer Datei und speichern des Inhalts in den
	 * Benutzereinstellungen.
	 *
	 * @return Ergebnis des Imports
	 */
	private IStatus dateiImportieren(final SpeicherKey einstellungsArt, final String fileName,
			final UrlasserInfo urlasser) {
		final ResourceSet resSet = new ResourceSetImpl();
		final File file = new File(fileName);
		final Resource resource = resSet.createResource(URI.createURI(file.toURI().toString()));
		try {
			resource.load(null);
		} catch (final IOException ex) {
			final String message = "Die Datei\n" + file + "\nenthält mind. einen Fehler."
					+ "\n\nMöchten Sie die Darstellung trotz der Fehler importieren?\n\nDabei "
					+ "würden die fehlenden Darstellungsobjekte aus der Darstellung entfernt. "
					+ "Andernfalls installieren Sie bitte zunächst die notwendigen Darstellungsobjekte!";

			final List<IStatus> errors = readErrors(resource);

			final MultiStatus multiStatus = new MultiStatus(DObjPlugin.PLUGIN_ID, IStatus.ERROR, "", null);
			errors.stream().forEach(multiStatus::add);

			final QuestionErrorDialog dialog = new QuestionErrorDialog(getShell(), "Fehler beim Import", message,
					multiStatus, IStatus.OK | IStatus.INFO | IStatus.WARNING | IStatus.ERROR);
			if (IDialogConstants.CANCEL_ID == dialog.open()) {
				return Status.error(message, ex);
			}
		}

		// Dateiinhalt prüfen
		final EList<EObject> contents = resource.getContents();
		if (contents.size() != 1) {
			final String message = "Die Datei ist fehlerhaft und kann nicht gelesen werden: " + file;
			final IStatus status = Status.error(message);

			ErrorDialog.openError(PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell(), "Fehler beim Import",
					message, status);
			return status;
		}
		final Named named = (Named) contents.get(0);

		named.eAllContents().forEachRemaining(e -> {
			if (EcoreUtil.getID(e) != null) {
				EcoreUtil.setID(e, EcoreUtil.generateUUID());
			}
		});

		if (EcoreUtil.getID(named) != null) {
			EcoreUtil.setID(named, EcoreUtil.generateUUID());
		}
		// Darstellung importieren
		try {
			if (einstellungen.getModellEinstellungen(einstellungsArt, named.getName()) != null) {
				if (MessageDialog.openQuestion(getShell(), getEClass().getInstanceTypeName() + " überschreiben",
						"Es existiert bereits ein Objekt mit dem selben Namen " + named.getName()
								+ ". Wollen Sie es überschreiben?")) {
					einstellungen.setModellEinstellungen(einstellungsArt, named.getName(), named, urlasser);
				} else {
					return Status.CANCEL_STATUS;
				}
			} else {
				einstellungen.setModellEinstellungen(einstellungsArt, named.getName(), named, urlasser);
			}
		} catch (final UrlasserDialogAbgebrochenException ex) {
			// Bediener hat den Urlasserdialog abgebrochen
			return Status.CANCEL_STATUS;
		} catch (final IOException ex) {
			final Status status = new Status(IStatus.ERROR, DObjPlugin.PLUGIN_ID, ex.getLocalizedMessage(), ex);
			ErrorDialog.openError(importPage.getShell(), "FEHLER",
					"Der Import konnte nicht erfolgreich ausgeführt werden!", status);
			return status;
		}
		return Status.OK_STATUS;
	}

	private List<IStatus> readErrors(final Resource resource) {
		final EList<Diagnostic> errors = resource.getErrors();

		final List<String> collect = errors.stream()
				.filter(d -> d instanceof org.eclipse.emf.ecore.xmi.PackageNotFoundException)
				.map(e -> (org.eclipse.emf.ecore.xmi.PackageNotFoundException) e).map(c -> c.uri()).distinct().toList();

		return collect.stream()
				.map(s -> new Status(IStatus.ERROR, DObjPlugin.PLUGIN_ID,
						"Das Darstellungsobjekt mit der Package-URI '" + s + "' ist nicht installiert."))
				.collect(Collectors.toList());
	}

}
