/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.wizards;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.emf.ecore.EClass;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ListViewer;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.views.properties.tabbed.ITabbedPropertyConstants;

import com.bitctrl.lib.eclipse.emf.viewers.EmfLabelProvider;

/**
 * Basisklasse für die Einstellungen zum Import eines in den
 * Rahmenwerkseinstellungen gespeicherten Einstellungen.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class AbstractImportWizardPage extends WizardPage {

	private static final int STANDARD_LABEL_WIDTH = 85;
	private static final int HSPACE = 5;
	private static final int VSPACE = 4;

	private final EClass eClass;
	private final List<String> fileNames;

	private ListViewer viewer;
	private Button addFileButton;
	private Button removeFileButton;

	/**
	 * Initialisiert die Wizard Page.
	 *
	 * @param eClass
	 *            der Typ der zu importierenden Objekte.
	 */
	protected AbstractImportWizardPage(final String pageName, final EClass eClass) {
		super(pageName);

		this.eClass = eClass;
		fileNames = new ArrayList<>();
		setPageComplete(false);
	}

	@Override
	public void createControl(final Composite parent) {
		final Composite container = new Composite(parent, SWT.NONE);
		container.setLayout(new FormLayout());
		FormData data;

		viewer = new ListViewer(container, SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL | SWT.BORDER);
		data = new FormData();
		data.left = new FormAttachment(0, STANDARD_LABEL_WIDTH);
		data.right = new FormAttachment(80, 0);
		data.top = new FormAttachment(0, VSPACE);
		data.height = 50;
		viewer.getControl().setLayoutData(data);
		viewer.setContentProvider(new ArrayContentProvider());
		viewer.setLabelProvider(new EmfLabelProvider());
		viewer.setInput(fileNames);
		viewer.addSelectionChangedListener(event -> removeFileButton.setEnabled(!event.getSelection().isEmpty()));

		final Label label = new Label(container, SWT.NONE);
		label.setText(eClass.getName());
		data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.right = new FormAttachment(viewer.getControl(), -HSPACE);
		data.top = new FormAttachment(viewer.getControl(), 0, SWT.TOP);
		label.setLayoutData(data);

		addFileButton = new Button(container, SWT.PUSH);
		addFileButton.setText("Hinzufügen");
		data = new FormData();
		data.left = new FormAttachment(viewer.getControl(), HSPACE);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(viewer.getControl(), 0, SWT.TOP);
		addFileButton.setLayoutData(data);
		addFileButton.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(final SelectionEvent e) {
				final FileDialog dlg = new FileDialog(getShell(), SWT.OPEN | SWT.MULTI);
				dlg.setFilterExtensions(new String[] { "*." + eClass.getName().toLowerCase(), "*.*" });
				dlg.setText(getTitle());
				if (dlg.open() != null) {
					final String path = dlg.getFilterPath();
					for (final String fileName : dlg.getFileNames()) {
						fileNames.add(path + "/" + fileName);
					}
				}
				viewer.refresh();
				updatePageComplete();
			}

		});

		removeFileButton = new Button(container, SWT.PUSH);
		removeFileButton.setText("Entfernen");
		data = new FormData();
		data.left = new FormAttachment(viewer.getControl(), HSPACE);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(addFileButton, ITabbedPropertyConstants.VSPACE);
		removeFileButton.setLayoutData(data);
		removeFileButton.setEnabled(false);
		removeFileButton.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(final SelectionEvent e) {
				fileNames.removeAll(((IStructuredSelection) viewer.getSelection()).toList());
				viewer.refresh();
				updatePageComplete();
			}

		});

		viewer.getControl().setFocus();
		setControl(container);
	}

	/**
	 * Gibt die Liste der zu importierenden (voll qualifizierten) Filenamen
	 * zurück.
	 */
	public List<String> getFileNames() {
		return fileNames;
	}

	private void updatePageComplete() {
		setPageComplete(!fileNames.isEmpty());
	}

}
