/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dobj.actions;

import org.eclipse.gef.SharedImages;
import org.eclipse.jface.action.IAction;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.actions.RetargetAction;

import de.bsvrz.buv.plugin.dobj.internal.DobjMessages;
import de.bsvrz.buv.plugin.dobj.util.DobjIcons;

/**
 * Factory für die {@link RetargetAction}s des Plug-ins. Die hier erzeugten
 * Actions tun nichts, sie agieren nur als Platzhalter. Der Workbench Part muss
 * eine Action bereitstellen mit der selben ID, die die tatsächliche Aktion
 * ausführt. Im Workbench Part muss diese Action mit
 * {@link IActionBars#setGlobalActionHandler(String, IAction)} bekannt gemacht
 * werden.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public abstract class DobjActionFactory {

	/**
	 * Schaltet auf das Tool zum Verschieben des aktuellen Ausschnitt um.
	 */
	public static final DobjActionFactory PAN = new DobjActionFactory("de.bsvrz.buv.plugin.dobj.command.pan") {

		@Override
		public RetargetAction create() {
			final RetargetAction action = new RetargetAction(getCommandId(), DobjMessages.panLabel,
					IAction.AS_CHECK_BOX);
			action.setToolTipText(DobjMessages.panTooltip);
			action.setImageDescriptor(DobjIcons.ActionPan.getImageDescriptor());
			action.setActionDefinitionId(getCommandId());
			return action;
		}
	};

	/**
	 * Schalten auf das Tool zur Selektion mittels eines Polygons um.
	 */
	public static final DobjActionFactory SELECTION_POLYGON = new DobjActionFactory(
			"de.bsvrz.buv.plugin.dobj.command.polygonSelection") {

		@Override
		public RetargetAction create() {
			final RetargetAction action = new RetargetAction(getCommandId(), "Polygonselektion", IAction.AS_CHECK_BOX);
			action.setToolTipText("Polygonselektion");
			action.setImageDescriptor(DobjIcons.ActionSelectionPolygon.getImageDescriptor());
			action.setActionDefinitionId(getCommandId());
			return action;
		}
	};

	/**
	 * Erzeugt eine Action zum Umschalten auf das Tool zur Selektion mittels
	 * eines Rechtecks.
	 */
	public static final DobjActionFactory SELECTION_RECTANGLE = new DobjActionFactory(
			"de.bsvrz.buv.plugin.dobj.command.rectangleSelection") {

		@Override
		public RetargetAction create() {
			final RetargetAction action = new RetargetAction(getCommandId(), DobjMessages.rectangleSelectionLabel,
					IAction.AS_CHECK_BOX);
			action.setToolTipText(DobjMessages.rectangleSelectionTooltip);
			action.setImageDescriptor(SharedImages.DESC_MARQUEE_TOOL_16);
			action.setActionDefinitionId(getCommandId());
			return action;
		}

	};

	/**
	 * Erzeugt eine Action zum Umschalten auf das Tool zur Selektion mittels
	 * eines Rechtecks.
	 */
	public static final DobjActionFactory SELECTION_ELLIPSE = new DobjActionFactory(
			"de.bsvrz.buv.plugin.dobj.command.ellipseSelection") {

		@Override
		public RetargetAction create() {
			final RetargetAction action = new RetargetAction(getCommandId(), DobjMessages.ellipseSelectionLabel,
					IAction.AS_CHECK_BOX);
			action.setToolTipText(DobjMessages.ellipseSelectionTooltip);
			action.setImageDescriptor(DobjIcons.ActionSelectionEllipse.getImageDescriptor());
			action.setActionDefinitionId(getCommandId());
			return action;
		}

	};

	/**
	 * Erzeugt eine Action zum Umschalten auf das Tool zur Selektion einer
	 * Strecke.
	 */
	public static final DobjActionFactory SELECTION_STRECKE = new DobjActionFactory(
			"de.bsvrz.buv.plugin.dobj.command.streckeSelection") {

		@Override
		public RetargetAction create() {
			final RetargetAction action = new RetargetAction(getCommandId(), DobjMessages.streckeSelectionLabel,
					IAction.AS_CHECK_BOX);
			action.setToolTipText(DobjMessages.streckeSelectionTooltip);
			action.setImageDescriptor(DobjIcons.ActionSelectionStrecke.getImageDescriptor());
			action.setActionDefinitionId(getCommandId());
			return action;
		}

	};

	/**
	 * Aktiviert und Deaktiviert die Teilselektion.
	 */
	public static final DobjActionFactory TOGGLE_TOUCHED_SELECTION = new DobjActionFactory(
			"de.bsvrz.buv.plugin.dobj.command.toogleTouchedSelection") {

		@Override
		public RetargetAction create() {
			final RetargetAction action = new RetargetAction(getCommandId(), DobjMessages.toggleTouchedSelectionLabel,
					IAction.AS_CHECK_BOX);
			action.setToolTipText(DobjMessages.toggleTouchedSelectionTooltip);
			action.setImageDescriptor(DobjIcons.ActionToggleTouchedSelection.getImageDescriptor());
			action.setActionDefinitionId(getCommandId());
			return action;
		}

	};

	/**
	 * Aktiviert und Deaktiviert das Beibehalten des Fensterausschnitts.
	 */
	public static final DobjActionFactory TOGGLE_FIX_VIEWPORT = new DobjActionFactory(
			"de.bsvrz.buv.plugin.dobj.command.toggleFixViewport") {

		@Override
		public RetargetAction create() {
			final RetargetAction action = new RetargetAction(getCommandId(), DobjMessages.toggleFixViewportLabel,
					IAction.AS_CHECK_BOX);
			action.setToolTipText(DobjMessages.toggleFixViewportTooltip);
			action.setImageDescriptor(DobjIcons.ActionToggleFixViewport.getImageDescriptor());
			action.setActionDefinitionId(getCommandId());
			return action;
		}

	};

	/**
	 * Aktiviert und Deaktiviert die Teilselektion.
	 */
	public static final DobjActionFactory TOGGLE_SELECTION_FEEDBACK = new DobjActionFactory(
			"de.bsvrz.buv.plugin.dobj.command.toogleSelectionFeedback") {

		@Override
		public RetargetAction create() {
			final RetargetAction action = new RetargetAction(getCommandId(), DobjMessages.toggleSelectionFeedbackLabel,
					IAction.AS_CHECK_BOX);
			action.setToolTipText(DobjMessages.toggleSelectionFeedbackTooltip);
			action.setImageDescriptor(DobjIcons.ActionToggleSelectionFeedback.getImageDescriptor());
			action.setActionDefinitionId(getCommandId());
			return action;
		}

	};

	/**
	 * Erzeugt eine Action die in einer Ansicht das Umschalten zwischen
	 * Onlinemodus und Offlinemodus erlaubt.
	 */
	public static final DobjActionFactory TOGGLE_ONLINE = new DobjActionFactory(
			"de.bsvrz.buv.plugin.dobj.command.toggleOnline") {

		@Override
		public RetargetAction create() {
			final RetargetAction action = new RetargetAction(getCommandId(), DobjMessages.toggleOnlineLabel,
					IAction.AS_CHECK_BOX);
			action.setToolTipText(DobjMessages.toggleOnlineTooltip);
			action.setImageDescriptor(DobjIcons.ActionToggleOnline.getImageDescriptor());
			action.setActionDefinitionId(getCommandId());
			return action;
		}

	};

	/**
	 * Erzeugt eine Action zum Umschalten auf das Tool zum Hineinzoomen in einen
	 * Ausschnitt.
	 */
	public static final DobjActionFactory ZOOM_RECTANGLE = new DobjActionFactory(
			"de.bsvrz.buv.plugin.dobj.command.zoomRectangle") {

		@Override
		public RetargetAction create() {
			final RetargetAction action = new RetargetAction("de.bsvrz.buv.plugin.dobj.command.zoomRectangle",
					DobjMessages.rectangleZoomLabel, IAction.AS_CHECK_BOX);
			action.setToolTipText(DobjMessages.rectangleZoomTooltip);
			action.setImageDescriptor(DobjIcons.ActionZoomRectangle.getImageDescriptor());
			action.setActionDefinitionId("de.bsvrz.buv.plugin.dobj.command.zoomRectangle");
			return action;
		}

	};

	/**
	 * Erzeugt eine Action die in einer Ansicht die Verbindungslinien zwischen
	 * einem verschobenen Objekt und seiner Originalposition ein- und
	 * auschaltet.
	 */
	public static final DobjActionFactory TOGGLE_VERBINDUNGSLINIE = new DobjActionFactory(
			"de.bsvrz.buv.plugin.dobj.command.toggleVerbindungslinie") {

		@Override
		public RetargetAction create() {
			final RetargetAction action = new RetargetAction(getCommandId(), DobjMessages.toggleVerbindungslinieLabel,
					IAction.AS_CHECK_BOX);
			action.setToolTipText(DobjMessages.toggleVerbindungslinieTooltip);
			action.setImageDescriptor(DobjIcons.ActionToggleVerbindingslinie.getImageDescriptor());
			action.setActionDefinitionId(getCommandId());
			return action;
		}

	};

	/**
	 * Erzeugt eine Action die in einer Ansicht die Verbindungslinien zwischen
	 * einem verschobenen Objekt und seiner Originalposition ein- und
	 * auschaltet.
	 */
	public static final DobjActionFactory TOGGLE_BAUSTELLE_GUELTIG = new DobjActionFactory(
			"de.bsvrz.buv.plugin.dobj.command.toggleBaustelleGueltig") {

		@Override
		public RetargetAction create() {
			final RetargetAction action = new RetargetAction(getCommandId(), DobjMessages.toggleBaustelleGueltigLabel,
					IAction.AS_CHECK_BOX);
			action.setToolTipText(DobjMessages.toggleBaustelleGueltigTooltip);
			action.setImageDescriptor(DobjIcons.ActionToggleBaustelleGueltig.getImageDescriptor());
			action.setActionDefinitionId(getCommandId());
			return action;
		}

	};

	/**
	 * Erzeugt eine Action die den Antikollisionsalgorithmus für
	 * Darstellunsobjekte ein- und auschaltet.
	 */
	public static final DobjActionFactory TOGGLE_ANTI_KOLLISIONS_ALGORITHMUS = new DobjActionFactory(
			"de.bsvrz.buv.plugin.dobj.command.toggleAntikollisionsAlgorithmus") {

		@Override
		public RetargetAction create() {
			final RetargetAction action = new RetargetAction(getCommandId(),
					DobjMessages.toggleAntiKollisionsAlgorithmusLabel, IAction.AS_CHECK_BOX);
			action.setToolTipText(DobjMessages.toggleAntiKollisionsAlgorithmusTooltip);
			action.setImageDescriptor(DobjIcons.ActionToggleAntikollisionsAlgorithmus.getImageDescriptor());
			action.setActionDefinitionId(getCommandId());
			return action;
		}

	};

	private final String commandId;

	private DobjActionFactory(final String commandId) {
		this.commandId = commandId;
	}

	/**
	 * Erzeugt eine neue {@link RetargetAction}.
	 */
	public abstract RetargetAction create();

	/**
	 * Gibt die (Command-)ID der {@link RetargetAction} zurück.
	 */
	public String getCommandId() {
		return commandId;
	}

}
