/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dobj.actions;

import java.util.List;

import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.PointList;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.GraphicalEditPart;
import org.eclipse.gef.GraphicalViewer;
import org.eclipse.gef.commands.Command;
import org.eclipse.gef.commands.CompoundCommand;
import org.eclipse.gef.requests.CreateRequest;
import org.eclipse.gef.requests.CreationFactory;
import org.eclipse.gef.ui.actions.Clipboard;
import org.eclipse.gef.ui.actions.SelectionAction;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.actions.ActionFactory;

import de.bsvrz.buv.plugin.dobj.model.DoModel;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * If the current object on the clipboard is a valid template, this action will
 * paste the template to the viewer.
 *
 * @author Eric Bordeau, Pratik Shah
 * @see org.eclipse.gef.ui.actions.CopyTemplateAction
 */
public class PasteDobjAction extends SelectionAction {

	private static final Debug LOGGER = Debug.getLogger();

	private static final class CloneCreationFactory implements CreationFactory {

		private final DoModel model;

		CloneCreationFactory(final DoModel model) {
			this.model = model;
		}

		@Override
		public Object getObjectType() {
			return model.getDoTyp();
		}

		@Override
		public Object getNewObject() {
			return EcoreUtil.copy(model);
		}

	}

	/**
	 * Constructor for PasteTemplateAction.
	 *
	 * @param editor
	 */
	public PasteDobjAction(final IWorkbenchPart editor) {
		super(editor);
	}

	/**
	 * @return <code>true</code> if {@link #createPasteCommand()} returns an
	 *         executable command
	 * @see org.eclipse.gef.ui.actions.WorkbenchPartAction#calculateEnabled()
	 */
	@Override
	protected boolean calculateEnabled() {
		return createPasteCommand() != null;
	}

	/**
	 * Creates and returns a command (which may be <code>null</code>) to create
	 * a new EditPart based on the template on the clipboard.
	 *
	 * @return the paste command
	 */
	protected Command createPasteCommand() {
		final CompoundCommand result = new CompoundCommand();
		final List<?> selection = getSelectedObjects();
		if (selection != null && selection.size() == 1) {
			final Object obj = selection.get(0);
			if (obj instanceof final GraphicalEditPart gep) {
				try {
					final List<DoModel> templates = (List<DoModel>) getClipboardContents();
					if (templates != null) {
						for (final DoModel template : templates) {
							final CreationFactory factory = getFactory(template);
							if (factory != null) {
								final CreateRequest request = new CreateRequest();
								request.setFactory(factory);
								request.setLocation(getPasteLocation(gep, templates, template));
								EditPart targetEditPart = gep.getTargetEditPart(request);
								EditPart editpart = gep;

								do {
									if (targetEditPart != null) {
										final Command cmd = targetEditPart.getCommand(request);
										if (cmd != null) {
											result.add(cmd);
											break;
										}
									}
									editpart = editpart.getParent();
									if (editpart != null) {
										targetEditPart = editpart.getTargetEditPart(request);
									}

								} while (targetEditPart != null && editpart != null);

							}
						}
					}
				} catch (final Exception ex) {
					// Das Clipboard kann potentiell andere Daten oder nicht
					// interpretierbare Daten enthalten
					// siehe NERZ FM-227
					LOGGER.warning("Fehler beim Auslesen des Clipboards: " + ex.getLocalizedMessage());
				}
			}
		}
		return result;
	}

	/**
	 * Returns the template on the clipboard, if there is one. Note that the
	 * template on the clipboard might be from a palette from another type of
	 * editor.
	 *
	 * @return the clipboard's contents
	 */
	protected Object getClipboardContents() {
		return Clipboard.getDefault().getContents();
	}

	/**
	 * Returns the appropriate Factory object to be used for the specified
	 * template. This Factory is used on the CreateRequest that is sent to the
	 * target EditPart. Note that the given template might be from a palette for
	 * a different GEF-based editor. In that case, this method can return
	 * <code>null</code>.
	 *
	 * @param template
	 *            the template Object; it will never be <code>null</code>
	 * @return a Factory
	 */
	protected CreationFactory getFactory(final Object template) {
		if (template instanceof DoModel) {
			return new CloneCreationFactory((DoModel) template);
		}
		return null;
	}

	/**
	 * @param container
	 *            the parent of the new part that is being pasted
	 * @param templates
	 * @param template
	 * @return the location at which
	 */
	protected Point getPasteLocation(final GraphicalEditPart container, final List<DoModel> templates,
			final DoModel template) {
		final GraphicalViewer viewer = getWorkbenchPart().getAdapter(GraphicalViewer.class);
		final Control control = viewer.getControl();
		final Point mouse = new Point(control.toControl(control.getDisplay().getCursorLocation()));

		final Point result;
		if (mouse.x >= 0 && mouse.y >= 0) {
			final PointList points = new PointList();
			for (final DoModel m : templates) {
				// TODO Was ist mit relativen Positionen??
				if (m.getLocation() != null) {
					points.addPoint(m.getLocation());
				}
			}
			final Rectangle bounds = points.getBounds();

			final Point p = template.getLocation();
			final int dx = p.x - bounds.x;
			final int dy = p.y - bounds.y;
			result = new Point(mouse.x + dx, mouse.y + dy);
		} else {
			result = new Point(10, 10);
		}

		return result;
	}

	@Override
	protected void init() {
		setId(ActionFactory.PASTE.getId());
		setText("Einfügen");
	}

	/**
	 * Executes the command returned by {@link #createPasteCommand()}.
	 */
	@Override
	public void run() {
		execute(createPasteCommand());
	}

}
