/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dobj.decorator.internal.properties.tabbed;

import static org.eclipse.swt.SWT.CENTER;
import static org.eclipse.ui.views.properties.tabbed.ITabbedPropertyConstants.HSPACE;
import static org.eclipse.ui.views.properties.tabbed.ITabbedPropertyConstants.VMARGIN;
import static org.eclipse.ui.views.properties.tabbed.ITabbedPropertyConstants.VSPACE;

import java.util.HashMap;
import java.util.Map;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.ColorDialog;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetWidgetFactory;

import com.bitctrl.lib.eclipse.emf.gef.commands.SetCommand;
import com.bitctrl.lib.eclipse.emf.gef.commands.UnsetCommand;

import de.bsvrz.buv.plugin.dobj.properties.AbstractSection;

abstract class ZustandsfarbenDecoratorPropertySection<T extends EObject> extends AbstractSection<T> {

	private final Map<EStructuralFeature, Label> labels = new HashMap<>();
	private final Map<EStructuralFeature, Text> colorTexts = new HashMap<>();
	private final Map<EStructuralFeature, Button> selectColorButtons = new HashMap<>();
	private final Map<EStructuralFeature, Button> useDefaultButtons = new HashMap<>();
	private final Map<EStructuralFeature, Color> colors = new HashMap<>();

	protected final Text createDatenstatusColor(final String zustand, final EStructuralFeature feature,
			final Composite parent, final Control vorgaenger) {
		// Widgets anlegen
		final TabbedPropertySheetWidgetFactory factory = getWidgetFactory();
		final Label label = factory.createLabel(parent, zustand);
		final Text colorText = factory.createText(parent, "", SWT.READ_ONLY);
		colorText.setEditable(false);
		colorText.setEnabled(false);
		final Button selectColorButton = factory.createButton(parent, "Ändern", SWT.PUSH);
		final Button useDefaultButton = factory.createButton(parent, "Defaultwert verwenden", SWT.CHECK);

		// Widgets layouten
		final FormData textData = new FormData();
		textData.left = new FormAttachment(label, HSPACE);
		textData.width = 150;
		if (vorgaenger != null) {
			textData.top = new FormAttachment(vorgaenger, VSPACE);
		} else {
			textData.top = new FormAttachment(0, VMARGIN);
		}
		colorText.setLayoutData(textData);
		final FormData formData = new FormData();
		formData.left = new FormAttachment(0, 0);
		formData.width = STANDARD_LABEL_WIDTH;
		formData.top = new FormAttachment(colorText, 0, CENTER);
		label.setLayoutData(formData);
		final FormData selectColorData = new FormData();
		selectColorData.left = new FormAttachment(colorText, HSPACE);
		selectColorData.top = new FormAttachment(colorText, 0, CENTER);
		selectColorButton.setLayoutData(selectColorData);
		final FormData useDefaultData = new FormData();
		useDefaultData.left = new FormAttachment(selectColorButton, 0, HSPACE);
		useDefaultData.top = new FormAttachment(colorText, 0, CENTER);
		useDefaultButton.setLayoutData(useDefaultData);

		// Listener anmelden
		selectColorButton.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(final SelectionEvent e) {
				final ColorDialog dlg = new ColorDialog(parent.getShell());
				dlg.setText("Farbe für den Zustand \"" + zustand + "\" auswählen");
				dlg.setRGB((RGB) getElement().eGet(feature));
				final RGB color = dlg.open();
				if (color != null) {
					final SetCommand command = new SetCommand(getElement(), feature, color);
					getCommandStack().execute(command);
				}
			}

		});
		useDefaultButton.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(final SelectionEvent e) {
				if (useDefaultButton.getSelection()) {
					final UnsetCommand cmd = new UnsetCommand(getElement(), feature);
					getCommandStack().execute(cmd);
				} else {
					final RGB value = (RGB) getElement().eGet(feature);
					final SetCommand cmd = new SetCommand(getElement(), feature, value);
					getCommandStack().execute(cmd);
				}
			}

		});

		labels.put(feature, label);
		colorTexts.put(feature, colorText);
		selectColorButtons.put(feature, selectColorButton);
		useDefaultButtons.put(feature, useDefaultButton);
		return colorText;
	}

	@Override
	public void refresh() {
		disposeColors();
		setColors();
	}

	private void disposeColors() {
		for (final Color c : colors.values()) {
			c.dispose();
		}
	}

	private void setColors() {
		for (final EStructuralFeature feature : colorTexts.keySet()) {
			final Color color = setColor(feature);
			colors.put(feature, color);
		}
	}

	private Color setColor(final EStructuralFeature feature) {
		final Label label = labels.get(feature);
		final Text text = colorTexts.get(feature);
		final Button selectColorButton = selectColorButtons.get(feature);
		final Button useDefaultButton = useDefaultButtons.get(feature);

		final RGB rgb = (RGB) getElement().eGet(feature);

		final Color color = new Color(text.getDisplay(), rgb);
		text.setText(rgb.toString());
		text.setBackground(color);

		final boolean useDefault = !getElement().eIsSet(feature);
		useDefaultButton.setSelection(useDefault);

		final boolean enabled = !useDefault;
		label.setEnabled(enabled);
		selectColorButton.setEnabled(enabled);

		return color;
	}

	@Override
	public void dispose() {
		disposeColors();
		super.dispose();
	}

}