/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dobj.editors;

import org.eclipse.core.runtime.Assert;
import org.eclipse.gef.ContextMenuProvider;
import org.eclipse.gef.EditPartViewer;
import org.eclipse.gef.ui.actions.ActionRegistry;
import org.eclipse.gef.ui.actions.GEFActionConstants;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.ui.actions.ActionFactory;

/**
 * Basisklasse für Context Menu Provider für Editoren die Darstellungsobjekte
 * und Darstellungen bearbeiten.
 *
 * <p>
 * Registriert Aktionen für UNDO, REDO und DELETE.
 *
 * <p>
 * Das Context Menu enthält eine Sektion für Erweiterungen.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class DobjEditorContextMenuProvider extends ContextMenuProvider {

	private final ActionRegistry actionRegistry;

	/**
	 * Initialisiert den Context Menu Provider.
	 *
	 * @param viewer
	 *            der Viewer für den Das Context Menu bestimmt ist.
	 * @param actionRegistry
	 *            die Action Registry.
	 */
	public DobjEditorContextMenuProvider(final EditPartViewer viewer, final ActionRegistry actionRegistry) {
		super(viewer);
		Assert.isNotNull(actionRegistry);
		this.actionRegistry = actionRegistry;

		setRemoveAllWhenShown(true);
	}

	@Override
	public void buildContextMenu(final IMenuManager menu) {
		// Add standard action groups to the menu
		GEFActionConstants.addStandardActionGroups(menu);

		// Add actions to the menu
		menu.appendToGroup(GEFActionConstants.GROUP_UNDO, // target group id
				getAction(ActionFactory.UNDO.getId())); // action to add
		menu.appendToGroup(GEFActionConstants.GROUP_UNDO, getAction(ActionFactory.REDO.getId()));

		// TODO menu.appendToGroup(GEFActionConstants.GROUP_EDIT,
		// getAction(ActionFactory.CUT.getId()));
		menu.appendToGroup(GEFActionConstants.GROUP_EDIT, getAction(ActionFactory.COPY.getId()));
		menu.appendToGroup(GEFActionConstants.GROUP_EDIT, getAction(ActionFactory.PASTE.getId()));
		menu.appendToGroup(GEFActionConstants.GROUP_EDIT, getAction(ActionFactory.DELETE.getId()));

		IAction action = getAction(GEFActionConstants.DIRECT_EDIT);
		if (action.isEnabled()) {
			menu.appendToGroup(GEFActionConstants.GROUP_EDIT, action);
		}

		// Alignment Actions
		final MenuManager submenu = new MenuManager("Ausrichtung");

		action = getAction(GEFActionConstants.ALIGN_LEFT);
		if (action.isEnabled()) {
			submenu.add(action);
		}

		action = getAction(GEFActionConstants.ALIGN_CENTER);
		if (action.isEnabled()) {
			submenu.add(action);
		}

		action = getAction(GEFActionConstants.ALIGN_RIGHT);
		if (action.isEnabled()) {
			submenu.add(action);
		}

		submenu.add(new Separator());

		action = getAction(GEFActionConstants.ALIGN_TOP);
		if (action.isEnabled()) {
			submenu.add(action);
		}

		action = getAction(GEFActionConstants.ALIGN_MIDDLE);
		if (action.isEnabled()) {
			submenu.add(action);
		}

		action = getAction(GEFActionConstants.ALIGN_BOTTOM);
		if (action.isEnabled()) {
			submenu.add(action);
		}

		if (!submenu.isEmpty()) {
			menu.appendToGroup(GEFActionConstants.GROUP_REST, submenu);
		}
	}

	private IAction getAction(final String actionId) {
		return actionRegistry.getAction(actionId);
	}

}
