/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dobj.editors;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Arrays;
import java.util.EventObject;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.draw2d.ColorConstants;
import org.eclipse.draw2d.ConnectionLayer;
import org.eclipse.draw2d.FigureCanvas;
import org.eclipse.draw2d.Layer;
import org.eclipse.draw2d.LayeredPane;
import org.eclipse.draw2d.LightweightSystem;
import org.eclipse.draw2d.MarginBorder;
import org.eclipse.draw2d.StackLayout;
import org.eclipse.draw2d.Viewport;
import org.eclipse.draw2d.geometry.Dimension;
import org.eclipse.draw2d.zoom.AbstractZoomManager;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.gef.ContextMenuProvider;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.EditPartFactory;
import org.eclipse.gef.GraphicalViewer;
import org.eclipse.gef.KeyHandler;
import org.eclipse.gef.KeyStroke;
import org.eclipse.gef.LayerConstants;
import org.eclipse.gef.MouseWheelHandler;
import org.eclipse.gef.MouseWheelZoomHandler;
import org.eclipse.gef.dnd.TemplateTransferDropTargetListener;
import org.eclipse.gef.editparts.ScalableRootEditPart;
import org.eclipse.gef.editparts.ZoomManager;
import org.eclipse.gef.ui.actions.ActionRegistry;
import org.eclipse.gef.ui.actions.DirectEditAction;
import org.eclipse.gef.ui.actions.GEFActionConstants;
import org.eclipse.gef.ui.actions.ToggleGridAction;
import org.eclipse.gef.ui.actions.ZoomInAction;
import org.eclipse.gef.ui.actions.ZoomOutAction;
import org.eclipse.gef.ui.parts.GraphicalEditor;
import org.eclipse.gef.ui.parts.GraphicalViewerKeyHandler;
import org.eclipse.gef.ui.parts.ScrollingGraphicalViewer;
import org.eclipse.gef.ui.parts.SelectionSynchronizer;
import org.eclipse.gef.ui.parts.TreeViewer;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.util.TransferDropTargetListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Canvas;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.actions.ActionFactory;
import org.eclipse.ui.part.IPageSite;
import org.eclipse.ui.part.Page;
import org.eclipse.ui.views.contentoutline.IContentOutlinePage;
import org.eclipse.ui.views.properties.IPropertySheetPage;
import org.eclipse.ui.views.properties.tabbed.ITabbedPropertySheetPageContributor;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetPage;

import com.bitctrl.lib.eclipse.draw2d.parts.ScrollableThumbnail2;
import com.bitctrl.lib.eclipse.draw2d.parts.Thumbnail2;
import com.bitctrl.lib.eclipse.emf.eclipse.model.Named;
import com.bitctrl.lib.eclipse.emf.util.EmfUtil;

import de.bsvrz.buv.plugin.dobj.actions.PanAction;
import de.bsvrz.buv.plugin.dobj.actions.RectangleZoomAction;
import de.bsvrz.buv.plugin.dobj.actions.SelectAllAction;
import de.bsvrz.buv.plugin.dobj.internal.RahmenwerkService;
import de.bsvrz.buv.plugin.dobj.requests.OnlineRequest;
import de.bsvrz.buv.plugin.dobj.tools.DObjSelectionTool;
import de.bsvrz.buv.plugin.dobj.util.DobjUtil;
import de.bsvrz.sys.funclib.bitctrl.daf.DavProvider;

/**
 * Basisklasse für Editoren die zum Bearbeiten von Darstellungsobjekten und
 * Darstellungen verwendet werden. Die Benutzung der Outline-View und
 * Properties-View (Tabbed Properties) ist vorbereitet.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 * @param <T>
 *            der Typ der zu bearbeitenden Objekte.
 */
public abstract class DobjGraphicalEditor<T extends Named> extends GraphicalEditor
		implements ITabbedPropertySheetPageContributor {

	private VerbundenPropertyChangeListener verbundenPropertyChangeListener;
	private DavProvider davProvider;
	private boolean online;

	private KeyHandler sharedKeyHandler;
	private T model;
	private final Class<T> modelClass;
	private Image titleImage;
	private SelectionSynchronizer synchronizer;

	/**
	 * Initialisiert den Editor.
	 *
	 * @param modelClass
	 *            die Klasse der zu bearbeitenden Objekte.
	 */
	protected DobjGraphicalEditor(final Class<T> modelClass) {
		this.modelClass = modelClass;
		setEditDomain(new DobjEditDomain(this));
	}

	@Override
	public void init(final IEditorSite site, final IEditorInput input) throws PartInitException {
		super.init(site, input);

		davProvider = RahmenwerkService.getService().getObjektFactory();
		online = davProvider.isVerbunden();

		verbundenPropertyChangeListener = new VerbundenPropertyChangeListener();
		davProvider.addPropertyChangeListener(DavProvider.PROP_VERBUNDEN, verbundenPropertyChangeListener);
	}

	private void updateOnline() {
		final Shell shell = getSite().getWorkbenchWindow().getShell();
		if (shell != null) {
			shell.getDisplay().asyncExec(() -> {
				if (online) {
					geheOnline();
				} else {
					geheOffline();
				}
			});
		}
	}

	@Override
	protected void createGraphicalViewer(final Composite parent) {
		final GraphicalViewer viewer = new ScrollingGraphicalViewer();
		viewer.createControl(parent);
		setGraphicalViewer(viewer);
		configureGraphicalViewer();
		hookGraphicalViewer();
		initializeGraphicalViewer();
	}

	/**
	 * Flag ob die View online geschalten ist. Dies ist nur dann der Fall, wenn
	 * eine Verbindung zum Datenverteiler besteht.
	 *
	 * @return <code>true</code>, wenn eine Datenverteilerverbindung besteht,
	 *         andernfalls <code>false</code>.
	 */
	public boolean isOnline() {
		return online;
	}

	/**
	 * Wird aufgerufen, wenn die Verbindung zum Datenverteiler hergestellt
	 * wurde.
	 */
	protected void geheOnline() {
		getGraphicalViewer().getControl().setEnabled(true);
		setContentDescription("");

		for (final EditPart editPart : DobjUtil.getAllChildren(getGraphicalViewer().getRootEditPart())) {
			editPart.performRequest(new OnlineRequest(true));
		}
	}

	/**
	 * Wird aufgerufen, wenn die Verbindung zum Datenverteiler getrennt wurde.
	 */
	protected void geheOffline() {
		final Control control = getGraphicalViewer().getControl();
		if (control != null && !control.isDisposed()) {
			control.setEnabled(false);
			setContentDescription("Keine Verbindung zum Datenverteiler.");

			for (final EditPart editPart : DobjUtil.getAllChildren(getGraphicalViewer().getRootEditPart())) {
				editPart.performRequest(new OnlineRequest(false));
			}
		}
	}

	/**
	 * Gibt das Objekt zurück, das bearbeitet wird.
	 *
	 * @return das hier zu bearbeitende Objekt.
	 */
	protected T getModel() {
		return model;
	}

	/**
	 * Gibt die Klasse der zu bearbeitenden Objekte zurück.
	 */
	protected Class<T> getModelClass() {
		return modelClass;
	}

	/**
	 * Holt aus dem Editor Input per Adapter für {@link #getModelClass()} das zu
	 * bearbeitende Objekt.
	 */
	@Override
	protected void setInput(final IEditorInput input) {
		super.setInput(input);

		model = EcoreUtil.copy(input.getAdapter(getModelClass()));
		setPartName(model.getName());

		final Object image = EmfUtil.getImage(model);
		if (image instanceof Image) {
			titleImage = (Image) image;
		} else if (image instanceof ImageDescriptor) {
			titleImage = ((ImageDescriptor) image).createImage();
		}
		if (titleImage != null) {
			setTitleImage(titleImage);
		}
	}

	@Override
	protected SelectionSynchronizer getSelectionSynchronizer() {
		if (synchronizer == null) {
			synchronizer = new SelectionSynchronizer();
		}
		return synchronizer;
	}

	@Override
	protected void configureGraphicalViewer() {
		super.configureGraphicalViewer();

		final ScalableRootEditPart rootEditPart = new ScalableRootEditPart() {

			@Override
			protected LayeredPane createPrintableLayers() {
				final LayeredPane pane = new LayeredPane();

				Layer layer = new ConnectionLayer();
				layer.setPreferredSize(new Dimension(5, 5));
				pane.add(layer, CONNECTION_LAYER);

				layer = new Layer();
				layer.setLayoutManager(new StackLayout());
				pane.add(layer, PRIMARY_LAYER);

				return pane;
			}

		};

		getGraphicalViewer().setRootEditPart(rootEditPart);
		getGraphicalViewer().setEditPartFactory(getEditPartFactory());
		getGraphicalViewer()
				.setKeyHandler(new GraphicalViewerKeyHandler(getGraphicalViewer()).setParent(getCommonKeyHandler()));

		configureZoomManager(rootEditPart.getZoomManager());

		final IAction zoomIn = new ZoomInAction(rootEditPart.getZoomManager());
		getActionRegistry().registerAction(zoomIn);
		final IAction zoomOut = new ZoomOutAction(rootEditPart.getZoomManager());
		getActionRegistry().registerAction(zoomOut);

		final ContextMenuProvider provider = createContextMenuProvider();
		getGraphicalViewer().setContextMenu(provider);
		getSite().registerContextMenu(getEditorSite().getId() + ".contextmenu", provider, getGraphicalViewer());

		loadProperties();

		// Hifstools
		final IAction showGrid = new ToggleGridAction(getGraphicalViewer());
		getActionRegistry().registerAction(showGrid);

		/* Dobj-Selection-Tool registrieren. */
		getGraphicalViewer().setProperty(DObjSelectionTool.class.getSimpleName(),
				((DobjEditDomain) getEditDomain()).getDobjSelectionTool());
	}

	@Override
	protected void initializeGraphicalViewer() {
		getGraphicalViewer().setContents(model);
	}

	/**
	 * Legt die Properties für den {@link GraphicalViewer} fest.
	 *
	 * Wird von {@link #configureGraphicalViewer()} aufgerufen.
	 */
	protected void loadProperties() {
		getGraphicalViewer().setProperty(MouseWheelHandler.KeyGenerator.getKey(SWT.MOD1),
				MouseWheelZoomHandler.SINGLETON);
	}

	/**
	 * Erzeugt einen neuen Context Menu Provider für den Editor.
	 */
	protected ContextMenuProvider createContextMenuProvider() {
		return new DobjContextMenuProvider(getGraphicalViewer(), getActionRegistry());
	}

	/**
	 * Gibt die Edit Part Factory für den Editor zurück.
	 */
	protected abstract EditPartFactory getEditPartFactory();

	/**
	 * Konfiguriert den Zoom Manager. Die Standardimplementierung ermöglicht die
	 * speziellen Zoomlevel FIT_ALL, FIT_HEIGHT und FIT_WIDTH.
	 *
	 * <p>
	 * Kann erweitert werden um eigene Zoomlevel.
	 */
	protected void configureZoomManager(final ZoomManager zoomManager) {
		zoomManager.setZoomAnimationStyle(AbstractZoomManager.ANIMATE_ZOOM_IN_OUT);
		zoomManager.setZoomLevelContributions(
				Arrays.asList(ZoomManager.FIT_ALL, ZoomManager.FIT_HEIGHT, ZoomManager.FIT_WIDTH));
	}

	@Override
	public void dispose() {
		geheOffline();

		if (titleImage != null) {
			titleImage.dispose();
		}

		getGraphicalViewer().setContents(null);
		davProvider.removePropertyChangeListener(DavProvider.PROP_VERBUNDEN, verbundenPropertyChangeListener);
		super.dispose();
	}

	/**
	 * Gibt die Outline Page für den Editor zurück.
	 */
	protected abstract IContentOutlinePage getOutlinePage();

	/**
	 * Gibt den Canvas des Graphical Viewers zurück.
	 */
	protected FigureCanvas getEditor() {
		return (FigureCanvas) getGraphicalViewer().getControl();
	}

	/**
	 * Aktiviert die Tastenkombinationen DEL für Löschen und F2 für Bearbeiten.
	 */
	protected KeyHandler getCommonKeyHandler() {
		if (sharedKeyHandler == null) {
			sharedKeyHandler = new KeyHandler();
			sharedKeyHandler.put(KeyStroke.getPressed(SWT.DEL, 127, 0),
					getActionRegistry().getAction(ActionFactory.DELETE.getId()));
			sharedKeyHandler.put(KeyStroke.getPressed(SWT.F2, 0),
					getActionRegistry().getAction(GEFActionConstants.DIRECT_EDIT));
		}
		return sharedKeyHandler;
	}

	/**
	 * Erfragt den Zoommanager des Graphical Viewers.
	 */
	protected ZoomManager getZoomManager() {
		return (ZoomManager) getGraphicalViewer().getProperty(ZoomManager.class.toString());
	}

	@Override
	protected void createActions() {
		super.createActions();

		final ActionRegistry registry = getActionRegistry();

		IAction action = new DirectEditAction(this);
		registry.registerAction(action);
		getSelectionActions().add(action.getId());

		action = new PanAction(this);
		registry.registerAction(action);

		action = new RectangleZoomAction(this);
		registry.registerAction(action);

		action = new SelectAllAction(this);
		registry.registerAction(action);
	}

	/**
	 * Setzt das Dirty-Flag des Editors.
	 */
	@Override
	public void commandStackChanged(final EventObject event) {
		firePropertyChange(IEditorPart.PROP_DIRTY);
		super.commandStackChanged(event);
	}

	@Override
	public Object getAdapter(final Class type) {
		if (type == DobjEditDomain.class) {
			return getEditDomain();
		} else if (type == IContentOutlinePage.class) {
			return getOutlinePage();
		} else if (type == IPropertySheetPage.class) {
			return new TabbedPropertySheetPage(this);
		} else if (type == ZoomManager.class) {
			return getZoomManager();
		} else if (type == getModelClass()) {
			return getModel();
		}

		return super.getAdapter(type);
	}

	/**
	 * Basisklasse für die Outline-View für Editoren zum Bearbeiten.
	 *
	 * <p>
	 * Es wird ein Kontextmenü mit der ID <code>{Editor-Id}.contextmenu</code>
	 * angelegt.
	 *
	 * @author BitCtrl Systems GmbH, Falko Schumann
	 */
	protected abstract class OutlinePage extends Page implements IContentOutlinePage, IAdaptable {

		private int maxBufferSize = 25600;
		private SashForm control;
		private TreeViewer viewer;
		private Canvas overview;
		private Thumbnail2 thumbnail;
		private DisposeListener disposeListener;

		/**
		 * Gibt die maximale Puffergröße für das Thumbnail zurück.
		 */
		public int getMaxBufferSize() {
			return maxBufferSize;
		}

		/**
		 * Legt die maximale Puffergröße für das Thumbnail fest.
		 */
		public void setMaxBufferSize(final int maxBufferSize) {
			this.maxBufferSize = maxBufferSize;
		}

		/**
		 * Registriert die Standardaktionen: UNDO, REDO und DELETE.
		 */
		@Override
		public void init(final IPageSite pageSite) {
			super.init(pageSite);
			final ActionRegistry registry = getActionRegistry();

			final IActionBars bars = pageSite.getActionBars();

			String id = ActionFactory.UNDO.getId();
			bars.setGlobalActionHandler(id, registry.getAction(id));

			id = ActionFactory.REDO.getId();
			bars.setGlobalActionHandler(id, registry.getAction(id));

			id = ActionFactory.DELETE.getId();
			bars.setGlobalActionHandler(id, registry.getAction(id));

			id = GEFActionConstants.ZOOM_IN;
			bars.setGlobalActionHandler(id, registry.getAction(id));

			id = GEFActionConstants.ZOOM_OUT;
			bars.setGlobalActionHandler(id, registry.getAction(id));

			bars.updateActionBars();
		}

		@Override
		public void createControl(final Composite parent) {
			control = new SashForm(parent, SWT.VERTICAL | SWT.SMOOTH);
			control.setBackground(ColorConstants.buttonDarker);

			final GridLayout layout = new GridLayout();
			layout.marginWidth = 0;
			layout.marginHeight = 0;
			layout.horizontalSpacing = 0;
			layout.verticalSpacing = 0;
			control.setLayout(layout);

			overview = new Canvas(control, SWT.NONE);
			GridDataFactory.fillDefaults().hint(SWT.DEFAULT, 150).applyTo(overview);

			final LightweightSystem lws = new LightweightSystem(overview);
			final ScalableRootEditPart root = (ScalableRootEditPart) getGraphicalViewer().getRootEditPart();
			thumbnail = new ScrollableThumbnail2(maxBufferSize, (Viewport) root.getFigure());
			thumbnail.setBorder(new MarginBorder(3));
			thumbnail.setSource(root.getLayer(LayerConstants.PRINTABLE_LAYERS));
			lws.setContents(thumbnail);
			disposeListener = e -> {
				if (thumbnail != null) {
					thumbnail.deactivate();
					thumbnail = null;
				}
			};
			getEditor().addDisposeListener(disposeListener);

			viewer = new TreeViewer();
			getTreeViewer().createControl(control);
			GridDataFactory.fillDefaults().grab(true, true).applyTo(viewer.getControl());

			configureOutlineViewer();
			hookOutlineViewer();
			initializeOutlineViewer();

			control.setWeights(30, 70);
		}

		@Override
		public void dispose() {
			unhookOutlineViewer();
			if (thumbnail != null) {
				thumbnail.deactivate();
				thumbnail = null;
			}

			super.dispose();
		}

		/**
		 * Wird aufgerufen, um den Outline Viewer zu aktivieren.
		 *
		 * Kann in abgeleiteten Klassen erweitert werden.
		 */
		protected void hookOutlineViewer() {
			getSelectionSynchronizer().addViewer(getTreeViewer());
		}

		/**
		 * Wird aufgerufen, um den Outline Viewer zu deaktivieren.
		 *
		 * Kann in abgeleiteten Klassen erweitert werden.
		 */
		protected void unhookOutlineViewer() {
			getSelectionSynchronizer().removeViewer(getTreeViewer());
			if (disposeListener != null && getEditor() != null && !getEditor().isDisposed()) {
				getEditor().removeDisposeListener(disposeListener);
			}
		}

		/**
		 * Konfiguriert den Outline Viewer.
		 *
		 * Kann in abgeleiteten Klassen erweitert werden.
		 */
		protected void configureOutlineViewer() {
			getTreeViewer().setEditDomain(getEditDomain());
			getTreeViewer().setEditPartFactory(getEditPartFactory());

			final ContextMenuProvider provider = createContextMenuProvider();
			getTreeViewer().setContextMenu(provider);
			getSite().registerContextMenu(getEditorSite().getId() + ".contextmenu", provider,
					getSite().getSelectionProvider());

			getTreeViewer().setKeyHandler(getCommonKeyHandler());
			getTreeViewer().addDropTargetListener(
					(TransferDropTargetListener) new TemplateTransferDropTargetListener(getTreeViewer()));
		}

		/**
		 * Initialisiert den Outline Viewer. Insbesondere wird der anzuzeigende
		 * Content gesetzt.
		 *
		 * Kann in abgeleiteten Klassen erweitert werden.
		 */
		protected void initializeOutlineViewer() {
			setContents(getModel());
			for (final TreeItem item : viewer.getControl().getItems()) {
				item.setExpanded(true);
				for (final TreeItem subItem : item.getItems()) {
					subItem.setExpanded(true);
				}
			}
		}

		/**
		 * Legt den anzuzeigenden Content fest.
		 *
		 * @param contents
		 *            ein Modellobjekt oder dessen Edit Part.
		 */
		protected void setContents(final Object contents) {
			getTreeViewer().setContents(contents);
		}

		@Override
		public Control getControl() {
			return control;
		}

		@Override
		public void setFocus() {
			viewer.getControl().setFocus();
		}

		/**
		 * Gibt den Outline Viewer zurück.
		 */
		protected TreeViewer getTreeViewer() {
			return viewer;
		}

		@Override
		public void addSelectionChangedListener(final ISelectionChangedListener listener) {
			getTreeViewer().addSelectionChangedListener(listener);
		}

		@Override
		public void removeSelectionChangedListener(final ISelectionChangedListener listener) {
			getTreeViewer().removeSelectionChangedListener(listener);
		}

		@Override
		public ISelection getSelection() {
			if (getTreeViewer() == null) {
				return StructuredSelection.EMPTY;
			}
			return getTreeViewer().getSelection();
		}

		@Override
		public void setSelection(final ISelection selection) {
			if (getTreeViewer() != null) {
				getTreeViewer().setSelection(selection);
			}
		}

		/**
		 * Gibt die Edit Part Factory für den Viewer der Outline zurück.
		 */
		protected abstract EditPartFactory getEditPartFactory();

		/**
		 * Erzeugt einen neuen Context Menu Provider für den Viewer der Outline.
		 */
		protected ContextMenuProvider createContextMenuProvider() {
			return new DobjContextMenuProvider(getTreeViewer(), getActionRegistry());
		}

		@Override
		public Object getAdapter(final Class type) {
			if (type == DobjEditDomain.class) {
				return getEditDomain();
			} else if (type == TreeViewer.class) {
				return getTreeViewer();
			} else if (type == ZoomManager.class) {
				return getGraphicalViewer().getProperty(ZoomManager.class.toString());
			} else if (type == getModelClass()) {
				return getModel();
			}

			return null;
		}

	}

	private final class VerbundenPropertyChangeListener implements PropertyChangeListener {
		@Override
		public void propertyChange(final PropertyChangeEvent evt) {
			online = (Boolean) evt.getNewValue();
			updateOnline();
		}

	}

}
