/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.dobj.editparts;

import org.eclipse.draw2d.IFigure;
import org.eclipse.draw2d.geometry.Dimension;
import org.eclipse.draw2d.geometry.PrecisionDimension;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.gef.EditPolicy;
import org.eclipse.gef.commands.Command;
import org.eclipse.gef.commands.CompoundCommand;
import org.eclipse.gef.editpolicies.ResizableEditPolicy;
import org.eclipse.gef.requests.ChangeBoundsRequest;

import com.bitctrl.lib.eclipse.emf.gef.model.GefPackage;

import de.bsvrz.buv.plugin.darstellung.commands.MoveDoModelCommand;
import de.bsvrz.buv.plugin.darstellung.commands.ResizeDoModelCommand;
import de.bsvrz.buv.plugin.dobj.decorator.ZoomVerhaltenFixMediator;
import de.bsvrz.buv.plugin.dobj.decorator.model.DobjDecoratorPackage;
import de.bsvrz.buv.plugin.dobj.figures.DoBildFigure;
import de.bsvrz.buv.plugin.dobj.model.DoBild;
import de.bsvrz.buv.plugin.dobj.model.DobjPackage;

/**
 * EditPart für ein Bild, das als Darstellungsobjekt behandelt wird.
 *
 * @author BitCtrl Systems GmbH, Christian Hoesel
 *
 */
public class DoBildEditPart extends DoModelEditPart<DoBild, DoBildFigure> {

	private final class BildResizableEditPolicy extends ResizableEditPolicy {
		@Override
		protected Command getMoveCommand(final ChangeBoundsRequest request) {
			return new MoveDoModelCommand(getModel(), getFigure().getHotspot().getCopy()
					.getTranslated(request.getMoveDelta().getScaled(1 / getZoomManager().getZoom())));
		}

		@Override
		protected Command getResizeCommand(final ChangeBoundsRequest request) {
			PrecisionDimension size = new PrecisionDimension(getFigure().getSize());
			size.performScale(getZoomManager().getZoom());

			final CompoundCommand command = new CompoundCommand("Bild Verschieben und Skalieren");

			if (size.isEmpty()) {
				size = new PrecisionDimension(100, 100);
			}

			if (getModel().isZoomVerhaltenFix()) {
				command.add(new ResizeDoModelCommand(getModel(), size.getExpanded(request.getSizeDelta())));
			} else {
				command.add(super.getResizeCommand(request));
			}

			command.add(getMoveCommand(request));
			return command;
		}
	}

	@Override
	protected DoBildFigure createFigure() {
		return new DoBildFigure(getResourceManager());
	}

	@Override
	protected IFigure getToolTip() {
		return null;
	}

	@Override
	protected void createEditPolicies() {
		super.createEditPolicies();

		installEditPolicy(EditPolicy.PRIMARY_DRAG_ROLE, new BildResizableEditPolicy());
		if (!isEditor()) {
			removeEditPolicy(EditPolicy.LAYOUT_ROLE);
		}
	}

	@Override
	public boolean isSelectable() {
		return isEditor() || super.isSelectable() && getModel().isSelektierbar();
	}

	@Override
	public void notifyChanged(final Notification notification) {
		super.notifyChanged(notification);
		if (DobjDecoratorPackage.Literals.ZOOM_VERHALTEN_FIX_DECORATOR__ZOOM_VERHALTEN_FIX.equals(
				notification.getFeature()) && notification.getNewBooleanValue() != notification.getOldBooleanValue()) {

			final double zoom = getZoomManager().getZoom();
			Dimension size = getModel().getSize();
			if (size == null) {
				size = new Dimension(100, 100);
			}
			if (notification.getNewBooleanValue()) {
				// ZoomFix von false auf true
				getModel().setSize(size.getScaled(zoom));
			} else {
				// ZoomFix von true auf false
				getModel().setSize(size.getScaled(1 / zoom));
			}
		} else if (GefPackage.Literals.SIZED__SIZE.equals(notification.getFeature())
				|| DobjPackage.Literals.DO_BILD__BILD.equals(notification.getFeature())) {
			refreshVisuals();
		}
	}

	@Override
	protected void refreshVisuals() {
		new ZoomVerhaltenFixMediator(this).mediate();
		if (getModel().getSize() != null) {
			getFigure().setSize(getModel().getSize().getCopy());
		} else {
			getFigure().setSize(new Dimension(100, 100));
		}
		if (getModel().getBild() != null) {
			getFigure().setBild(getModel().getBild());
		}
		super.refreshVisuals();
	}
}
