/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dobj.editparts;

import org.eclipse.gef.EditPart;
import org.eclipse.gef.EditPartFactory;

import de.bsvrz.buv.plugin.dobj.model.DoModel;
import de.bsvrz.buv.plugin.dobj.model.Verbindungslinie;

/**
 * Basisklasse für die Factory eines Viewer, der {@link DoModel}s anzeigen soll.
 *
 * Diese Factory arbeitet nur mit Modellobjekten die von der Klasse
 * {@link DoModel} abgeleitet sind. Für die konrekte Darstellung im Editor,
 * einer Karte usw. muss diese Factory um die zusätzlichen Modellobjekte (z. B.
 * <code>MassstaeblicheDarstellung</code>) erweitert werden.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class DoEditPartFactory implements EditPartFactory {

	/**
	 * Erzeugt das Edit Part und weist ihm das Modellobjekt zu.
	 *
	 * <p>
	 * Modellobjekte vom Typ {@link DoModel} behandelt die Methdode selbst, alle
	 * anderen Objekte werden an {@link #doCreateEditPart(EditPart, Object)}
	 * delegiert.
	 *
	 * @see #doCreateEditPart(EditPart, Object)
	 */
	@Override
	public final EditPart createEditPart(final EditPart context, final Object model) {
		final EditPart editPart;
		if (model instanceof final DoModel doModel) {
			editPart = doModel.getDoTyp().createController();
		} else if (model instanceof Verbindungslinie) {
			editPart = new VerbindungslinieEditPart();
		} else {
			editPart = doCreateEditPart(context, model);
		}

		if (editPart.getModel() == null) {
			editPart.setModel(model);
		}

		return editPart;
	}

	/**
	 * Erzeugt den Edit Part zu einem Modellobjekt, das kein {@link DoModel}
	 * ist.
	 *
	 * <p>
	 * Darf in abgeleiteten Klassen nur erweitert werden. Der Aufruf der
	 * Supermethode muss dabei als letztes erfolgen.
	 *
	 * @param context
	 *            der übergeordnete Edit Part.
	 * @param model
	 *            das Modellobjekt zu dem ein Edit Part gesucht ist.
	 * @return der Edit Part für das Modellobjekt, niemals <code>null</code>.
	 */
	protected EditPart doCreateEditPart(final EditPart context, final Object model) {
		throw new IllegalArgumentException("Unbekanntes Modellobjekt: " + model.getClass());
	}

}
