/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dobj.editparts;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.ISafeRunnable;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.SafeRunner;
import org.eclipse.core.runtime.Status;
import org.eclipse.draw2d.IFigure;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.PrecisionPoint;

import de.bsvrz.buv.plugin.dobj.DoFigure;
import de.bsvrz.buv.plugin.dobj.internal.DObjPlugin;
import de.bsvrz.buv.plugin.dobj.model.DoModel;
import de.bsvrz.buv.plugin.dobj.util.Projektion;
import de.bsvrz.sys.funclib.bitctrl.geolib.WGS84Koordinate;
import de.bsvrz.sys.funclib.bitctrl.geolib.WGS84Polygon;
import de.bsvrz.sys.funclib.bitctrl.geolib.WGS84Punkt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.objekte.Punkt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.objekte.PunktLiegtAufLinienObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.util.geolib.WGS84Util;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Basisklasse für Darstellungsobjekte die Objekte vom Typ
 * <code>typ.punkt</code> repräsentieren.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 * @param <T>
 */
public abstract class PunktEditPart<T extends Punkt, F extends IFigure> extends GeoRefererenzObjektEditPart<T, F> {

	private static final Debug LOGGER = Debug.getLogger();
	private Point wgs84Location;
	private Point location;

	@Override
	protected void cleanModellKoordinaten() {
		location = null;
	}

	/**
	 * Gibt die Weltkoordinaten (konvertiert nach WGS84) zurück wie sie aus der
	 * Konfiguration ermittelt wurden.
	 *
	 * <p>
	 * Kann mit <code>PunktLiegtAufLinienObjekt</code> und <code>PunktXY</code>
	 * umgehen. Wenn ein Systemobjekt beides Typen erweitert, dann wird zuerst
	 * versucht die Koordinaten per <code>PunktLiegtAufLinienObjekt</code> zu
	 * ermitteln, ist dies nicht möglich wird anschließend versucht die
	 * Koordinaten per <code>PunktXY</code> zu ermitteln.
	 *
	 * @return die konfigurierten WGS-84-Koordinaten des Systemobjekts oder
	 *         <code>null</code>, wenn keine konfiguriert sind.
	 */
	protected Point getWgs84Location() {
		if (wgs84Location == null) {
			final Punkt p = getModel().getSystemObjekt();
			try {
				final WGS84Koordinate wgs84 = WGS84Util.konvertiere(p);
				wgs84Location = new PrecisionPoint(wgs84.getLaenge(), wgs84.getBreite());
			} catch (final RuntimeException ex) {
				LOGGER.error("Die WGS-84-Koordinaten des Punkts konnten nicht bestimmt werden: " + p, ex);
			}
		}

		return wgs84Location;
	}

	/**
	 * Gibt die Modellkoordinaten (ggf. nach Transformation mit Hilfe der
	 * Projektion) zurück.
	 *
	 * @return die Koordinaten des Punkts in Modellkoordinaten oder
	 *         <code>null</code>, wenn keine konfiguriert sind.
	 * @see #getWgs84Location()
	 */
	protected Point getLocation() {
		if (location == null) {
			final Projektion projektion = getProjektion();
			final Point wgs84 = getWgs84Location();
			if (projektion != null && wgs84 != null) {
				location = projektion.umrechnenWeltNachModel(wgs84);
			} else {
				location = wgs84;
			}
		}

		return location;
	}

	/**
	 * Ist der Punkt vom Typ <code>PunktLiegtAufLinienObjekt</code>, dann
	 * liefert diese Methode den Streckenabschnitt des Straßensegments auf dem
	 * der Punkt liegt. Diese Methode ist nützlich, wenn die Figure des Edit
	 * Parts z.&nbsp;B. rechtwinkelig an der Linie auf dem der Punkt liegt
	 * ausgerichtet werden soll.
	 *
	 * @return zwei Punkt die den Streckenabschnitt beschreiben oder
	 *         <code>null</code>, wenn der Streckenabschnitt nicht existiert
	 *         oder nicht bestimmt werden kann. Der Streckenabschnitt beginnt
	 *         mit dem verlaengerten Lot dieses Punktes auf dem Strassensegment.
	 */
	protected Point[] getStreckenAbschnitt() {
		if (getProjektion() != null && getModel().getSystemObjekt() instanceof PunktLiegtAufLinienObjekt) {
			final PunktLiegtAufLinienObjekt p = (PunktLiegtAufLinienObjekt) getModel().getSystemObjekt();

			if (p != null && p.getKdPunktLiegtAufLinienObjekt() != null
					&& p.getKdPunktLiegtAufLinienObjekt().getDatum() != null
					&& p.getKdPunktLiegtAufLinienObjekt().getDatum().getOffset() != null) {

				final WGS84Polygon teilstrecke = WGS84Util.getLinie(p);
				if (teilstrecke.laenge() == 0) {
					return null;
				}

				final double l = p.getKdPunktLiegtAufLinienObjekt().getDatum().getOffset().doubleValue();

				final List<WGS84Punkt> pAbstandDummyPunkte = new ArrayList<>();
				final List<WGS84Punkt> punkteAufLinie = teilstrecke.getKoordinaten();
				double laengeBisJetzt = 0.0;
				if (punkteAufLinie.size() > 1) {
					for (int i = 0; i < punkteAufLinie.size() - 1; i++) {
						pAbstandDummyPunkte.clear();
						pAbstandDummyPunkte.add(punkteAufLinie.get(i));
						pAbstandDummyPunkte.add(punkteAufLinie.get(i + 1));
						final WGS84Polygon pAbstandDummy = new WGS84Polygon(pAbstandDummyPunkte);
						final double abstand = pAbstandDummy.laenge();

						if (l >= laengeBisJetzt && l <= laengeBisJetzt + abstand) {
							final double laengeAufAbschnitt = l - laengeBisJetzt;
							if (laengeAufAbschnitt > 0 && abstand > 0) {
								final WGS84Punkt wgsP1 = punkteAufLinie.get(i);
								final WGS84Punkt wgsP2 = punkteAufLinie.get(i + 1);
								final Point p1 = getProjektion().umrechnenWeltNachModel(
										new PrecisionPoint(wgsP1.getLaenge(), wgsP1.getBreite()));
								final Point p2 = getProjektion().umrechnenWeltNachModel(
										new PrecisionPoint(wgsP2.getLaenge(), wgsP2.getBreite()));
								final double xDiff = p2.preciseX() - p1.preciseX();
								final double yDiff = p2.preciseY() - p1.preciseY();
								final double verhaeltnis = laengeAufAbschnitt / abstand;
								final double p1XNeu = p1.preciseX() + xDiff * verhaeltnis;
								final double p1YNeu = p1.preciseY() + yDiff * verhaeltnis;
								final Point p1Neu = new Point((int) p1XNeu, (int) p1YNeu);
								if (!p1Neu.equals(p2)) {
									return new Point[] { p1Neu, p2 };
								}
							}
						}

						laengeBisJetzt += abstand;
					}
				}
			}

			/**
			 * Wenn bis hierhin kein Ergebnis, benutze alte Methode (Backup):
			 */
			try {
				final WGS84Koordinate[] teilstrecke = WGS84Util.findTeilstrecke(p);
				final Point[] g = new Point[2];

				g[0] = getProjektion().umrechnenWeltNachModel(
						new PrecisionPoint(teilstrecke[0].getLaenge(), teilstrecke[0].getBreite()));
				g[1] = getProjektion().umrechnenWeltNachModel(
						new PrecisionPoint(teilstrecke[1].getLaenge(), teilstrecke[1].getBreite()));

				return g;
			} catch (final RuntimeException ex) {
				// Wird ignoriert, es gibt dann keinen Streckenabschnitt
				LOGGER.fine("Der Streckenabschnitt auf dem der Punkt liegt kann nicht bestimmt werden", p);
			}
		}

		return null;
	}

	/**
	 * Wenn das {@link DoModel} keine Location besitzt und eine Projektion
	 * existiert, wird die Location der Figure mit Hilfe des projezierten Punkts
	 * positioniert.
	 */
	@Override
	protected void refreshVisuals() {
		super.refreshVisuals();

		if (getModel().getSystemObject() != null && getModel().getLocation() == null && getProjektion() != null) {
			SafeRunner.run(new ISafeRunnable() {

				@Override
				public void run() throws Exception {
					final IFigure f = getFigure();
					final Point p = getLocation();
					if (f instanceof final DoFigure doFigure) {
						doFigure.setHotspot(p);
					} else {
						f.setLocation(p);
					}
				}

				@Override
				public void handleException(final Throwable exception) {
					DObjPlugin.getDefault().getLog().log(new Status(IStatus.WARNING, DObjPlugin.PLUGIN_ID,
							"Bestimmung der Position für " + getModel() + " nicht möglich.", exception));
				}
			});

		}
	}

}
