/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dobj.editparts;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.draw2d.AbstractPointListShape;
import org.eclipse.draw2d.IFigure;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.PointList;
import org.eclipse.draw2d.geometry.PrecisionPoint;
import org.eclipse.gef.GraphicalEditPart;

import de.bsvrz.buv.plugin.dobj.util.Projektion;
import de.bsvrz.sys.funclib.bitctrl.geolib.WGS84Koordinate;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.objekte.GeoReferenzObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.objekte.Linie;
import de.bsvrz.sys.funclib.bitctrl.modell.util.geolib.WGS84Util;

/**
 * Basisklasse für Edit Parts die als Controller für Systemobjekte die
 * <code>typ.linie</code> oder <code>typ.fläche</code> erweitern.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 * @param <T>
 */
public abstract class PunktlisteEditPart<T extends GeoReferenzObjekt, F extends AbstractPointListShape>
		extends GeoRefererenzObjektEditPart<T, F> {

	private List<Point> wgs84Points;
	private PointList points;

	@Override
	protected void cleanModellKoordinaten() {
		points = null;
	}

	/**
	 * Bestimmt die konfigurierte Linie des Systemobjekts.
	 *
	 * @return die Bildschirmkoordinaten des Systemobjekts.
	 * @see WGS84Util#konvertiere(Linie)
	 */
	protected List<Point> getWgs84Points() {
		if (wgs84Points == null) {
			final List<WGS84Koordinate> koordinaten = doGetWgs84Koordinaten();
			wgs84Points = new ArrayList<>();
			for (final WGS84Koordinate p : koordinaten) {
				wgs84Points.add(new PrecisionPoint(p.getLaenge(), p.getBreite()));
			}
		}

		return wgs84Points;
	}

	/**
	 * Bestimmt die WGS-84-Koordinaten die als Grundlage zum Zeichnen verwendet
	 * werden sollen.
	 */
	protected abstract List<WGS84Koordinate> doGetWgs84Koordinaten();

	/**
	 * Bestimmt die konfigurierte Linie des Systemobjekts.
	 *
	 * @return die Bildschirmkoordinaten des Systemobjekts.
	 * @see WGS84Util#konvertiere(Linie)
	 */
	public PointList getPoints() {
		if (points == null) {
			points = new PointList();
			final Projektion aktuelleProjektion = getProjektion();
			for (final Point p : getWgs84Points()) {
				points.addPoint(
						aktuelleProjektion.umrechnenWeltNachModel(new PrecisionPoint(p.preciseX(), p.preciseY())));
			}
		}

		return points;
	}

	@Override
	protected void refreshVisuals() {
		super.refreshVisuals();

		if (getModel().getSystemObject() != null && getModel().getLocation() == null && getProjektion() != null) {
			final IFigure f = getFigure();
			getFigure().setPoints(getPoints());
			((GraphicalEditPart) getParent()).setLayoutConstraint(this, f, f.getBounds());
		}
	}

}
