/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.dobj.editparts;

import org.eclipse.draw2d.IFigure;
import org.eclipse.draw2d.PolylineConnection;
import org.eclipse.draw2d.zoom.ZoomListener;
import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.ecore.util.EContentAdapter;
import org.eclipse.gef.Request;
import org.eclipse.gef.editparts.AbstractConnectionEditPart;
import org.eclipse.gef.editparts.ZoomManager;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.RGB;

import de.bsvrz.buv.plugin.darstellung.editparts.EbeneEditPart;
import de.bsvrz.buv.plugin.darstellung.model.Darstellung;
import de.bsvrz.buv.plugin.darstellung.model.DarstellungPackage;
import de.bsvrz.buv.plugin.darstellung.model.DarstellungsSpalte;
import de.bsvrz.buv.plugin.darstellung.model.Ebene;
import de.bsvrz.buv.plugin.darstellung.model.StilisierteDarstellung;
import de.bsvrz.buv.plugin.dobj.decorator.model.LinienStyle;
import de.bsvrz.buv.plugin.dobj.model.DoModel;
import de.bsvrz.buv.plugin.dobj.model.DobjPackage;
import de.bsvrz.buv.plugin.dobj.model.Verbindungslinie;
import de.bsvrz.buv.plugin.dobj.requests.DobjRequestConstants;
import de.bsvrz.buv.plugin.dobj.requests.VerbindungslinieRequest;

/**
 * Dieser Edit Part verwaltet eine Verbindung zwischen zwei
 * Darstellungsobjekten.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 */
public class VerbindungslinieEditPart extends AbstractConnectionEditPart implements ZoomListener {

	private final Adapter darstellungAdapter = new DarstellungAdapter();
	private final Adapter verbindungslinieAdapter = new VerbindunglinienAdapter();
	private boolean verbindungslinieSichtbar = true;
	private IFigure ebenenFigure;

	@Override
	protected IFigure createFigure() {
		return new VerbindungslinieFigure();
	}

	@Override
	public PolylineConnection getFigure() {
		return (PolylineConnection) super.getFigure();
	}

	@Override
	public Verbindungslinie getModel() {
		return (Verbindungslinie) super.getModel();
	}

	@Override
	protected void activateFigure() {
		ebenenFigure = findLayer();
		if (getFigure().getParent() != ebenenFigure) {
			ebenenFigure.add(getFigure(), -1);
		}
	}

	private IFigure findLayer() {
		IFigure result = getLayer(CONNECTION_LAYER);
		EbeneEditPart targetParent = null;
		EbeneEditPart sourceParent = null;
		if (getTarget() != null && getTarget().getParent() instanceof final EbeneEditPart ebenePart) {
			targetParent = ebenePart;
			targetParent.getContentPane().add(getFigure());
		}

		if (getSource() != null && getSource().getParent() instanceof final EbeneEditPart ebenePart) {
			sourceParent = ebenePart;
		}

		if (targetParent != null) {
			result = targetParent.getContentPane();
		} else if (sourceParent != null) {
			result = sourceParent.getContentPane();
		}
		return result;
	}

	@Override
	protected void deactivateFigure() {
		ebenenFigure.remove(getFigure());
		getConnectionFigure().setSourceAnchor(null);
		getConnectionFigure().setTargetAnchor(null);
	}

	@Override
	protected void createEditPolicies() {
		// tut nix

	}

	@Override
	public boolean isSelectable() {
		return false;
	}

	@Override
	public void activate() {
		super.activate();

		final Darstellung d = getDarstellung();
		if (d != null) {
			d.eAdapters().add(darstellungAdapter);
		}
		getModel().eAdapters().add(verbindungslinieAdapter);

		getZoomManager().addZoomListener(this);
	}

	private Darstellung getDarstellung() {
		return (Darstellung) getViewer().getProperty(Darstellung.class.toString());
	}

	private ZoomManager getZoomManager() {
		return (ZoomManager) getViewer().getProperty(ZoomManager.class.toString());
	}

	@Override
	public void deactivate() {
		getZoomManager().removeZoomListener(this);
		getModel().eAdapters().remove(verbindungslinieAdapter);
		final Darstellung d = getDarstellung();
		if (d != null) {
			d.eAdapters().remove(darstellungAdapter);
		}

		super.deactivate();
	}

	@Override
	public void zoomChanged(final double zoom) {
		refreshLinienstaerke();
	}

	@Override
	protected void refreshVisuals() {
		super.refreshVisuals();
		refreshVisibility();
		refreshLinienfarbe();
		refreshLinienstaerke();
		refreshLinienstil();
	}

	private void refreshVisibility() {
		final Darstellung d = getDarstellung();
		final PolylineConnection f = getFigure();
		Ebene quelleEbene = null;
		DarstellungsSpalte quelleSpalte = null;
		Ebene zielEbene = null;
		DarstellungsSpalte zielSpalte = null;
		final DoModel quelle = getModel().getQuelle();
		final DoModel ziel = getModel().getZiel();
		for (final Ebene e : d.getEbenen()) {
			if (e.getDoObjekte().contains(quelle)) {
				quelleEbene = e;
			}
			if (e.getDoObjekte().contains(ziel)) {
				zielEbene = e;
			}
		}
		if (d instanceof final StilisierteDarstellung sd) {
			for (final DarstellungsSpalte s : sd.getSpalten()) {
				if (s.getDoObjekte().contains(quelle)) {
					quelleSpalte = s;
				}
				if (s.getDoObjekte().contains(ziel)) {
					zielSpalte = s;
				}
			}
		}
		final boolean quelleIsVisible = quelleEbene != null && quelleEbene.isVisible()
				&& (quelleSpalte == null || quelleSpalte.isVisible());
		final boolean zielIsVisible = zielEbene != null && zielEbene.isVisible()
				&& (zielSpalte == null || zielSpalte.isVisible());
		f.setVisible(verbindungslinieSichtbar && quelleIsVisible && zielIsVisible);
	}

	private void refreshLinienstaerke() {
		final Darstellung d = getDarstellung();
		final PolylineConnection f = getFigure();
		final double normalizedLineWidth = d.getVerbindungslinieStaerke() / getZoomManager().getZoom();
		f.setLineWidthFloat((float) normalizedLineWidth);
	}

	private void refreshLinienfarbe() {
		final Darstellung d = getDarstellung();
		final PolylineConnection f = getFigure();
		final RGB rgb = d.getVerbindungslinieFarbe();
		final Color color = getViewer().getResourceManager().createColor(rgb);
		f.setForegroundColor(color);
	}

	private void refreshLinienstil() {
		final Darstellung d = getDarstellung();
		final PolylineConnection f = getFigure();

		final LinienStyle stil = d.getVerbindungslinieStil();
		switch (stil) {
		case DURCHGEZOGEN:
			f.setLineStyle(SWT.LINE_SOLID);
			break;
		case PUNKT:
			f.setLineStyle(SWT.LINE_DOT);
			break;
		case STRICH:
			f.setLineStyle(SWT.LINE_DASH);
			break;
		case STRICH_PUNKT:
			f.setLineStyle(SWT.LINE_DASHDOT);
			break;
		case STRICH_PUNKT_PUNKT:
			f.setLineStyle(SWT.LINE_DASHDOTDOT);
			break;
		default:
			throw new IllegalStateException("Unbekannter Enum-Wert für " + LinienStyle.class + ": " + stil);
		}
	}

	@Override
	public void performRequest(final Request req) {
		super.performRequest(req);

		if (DobjRequestConstants.REQ_VERBINDUNGSLINIE.equals(req.getType())) {
			final VerbindungslinieRequest vlr = (VerbindungslinieRequest) req;
			verbindungslinieSichtbar = vlr.isVerbindungsLinie();
			refreshVisibility();
		}
	}

	private final class VerbindunglinienAdapter extends EContentAdapter {
		@Override
		public void notifyChanged(final Notification msg) {
			super.notifyChanged(msg);
			if (msg.getNotifier() instanceof Verbindungslinie) {
				switch (msg.getEventType()) {
				case Notification.SET, Notification.UNSET:
					switch (msg.getFeatureID(Verbindungslinie.class)) {
					case DobjPackage.VERBINDUNGSLINIE__QUELLE, DobjPackage.VERBINDUNGSLINIE__ZIEL:
						refreshVisibility();
						break;
					default:
						// alle anderen Features interessieren uns hier nicht
						break;
					}
					break;
				default:
					// alle anderen Events interessiereun uns hier nicht
					break;
				}
			}
		}
	}

	private final class DarstellungAdapter extends EContentAdapter {

		@Override
		public void notifyChanged(final Notification msg) {
			handleDarstellung(msg);
			handleEbene(msg);
			handleDarstellungsspalte(msg);
		}

		private void handleDarstellung(final Notification msg) {
			if (msg.getNotifier() instanceof Darstellung) {
				switch (msg.getEventType()) {
				case Notification.SET, Notification.UNSET:
					switch (msg.getFeatureID(Darstellung.class)) {
					case DarstellungPackage.DARSTELLUNG__VERBINDUNGSLINIE_FARBE:
						refreshLinienfarbe();
						break;
					case DarstellungPackage.DARSTELLUNG__VERBINDUNGSLINIE_STAERKE:
						refreshLinienstaerke();
						break;
					case DarstellungPackage.DARSTELLUNG__VERBINDUNGSLINIE_STIL:
						refreshLinienstil();
						break;
					default:
						// alle anderen Features interessieren uns hier nicht
						break;
					}
					break;
				default:
					// alle anderen Events interessiereun uns hier nicht
					break;
				}
			}
		}

		private void handleEbene(final Notification msg) {
			if (msg.getNotifier() instanceof Ebene) {
				switch (msg.getEventType()) {
				case Notification.SET, Notification.UNSET:
					switch (msg.getFeatureID(Darstellung.class)) {
					case DarstellungPackage.EBENE__VISIBLE:
						refreshVisibility();
						break;
					default:
						// alle anderen Features interessieren uns hier nicht
						break;
					}
					break;
				default:
					// alle anderen Events interessiereun uns hier nicht
					break;
				}
			}
		}

		private void handleDarstellungsspalte(final Notification msg) {
			if (msg.getNotifier() instanceof DarstellungsSpalte) {
				switch (msg.getEventType()) {
				case Notification.SET, Notification.UNSET:
					switch (msg.getFeatureID(Darstellung.class)) {
					case DarstellungPackage.DARSTELLUNGS_SPALTE__VISIBLE:
						refreshVisibility();
						break;
					default:
						// alle anderen Features interessieren uns hier nicht
						break;
					}
					break;
				default:
					// alle anderen Events interessiereun uns hier nicht
					break;
				}
			}
		}
	}

	private static final class VerbindungslinieFigure extends PolylineConnection {

		// {
		// // Defaultwerte setzen
		// setForegroundColor(ColorConstants.lightGray);
		// setLineWidth(1);
		// setLineStyle(SWT.LINE_DASH);
		// }

	}

}
