/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dobj.editpolicies;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.draw2d.ColorConstants;
import org.eclipse.draw2d.Cursors;
import org.eclipse.draw2d.Graphics;
import org.eclipse.draw2d.geometry.PointList;
import org.eclipse.draw2d.geometry.PrecisionPoint;
import org.eclipse.gef.DragTracker;
import org.eclipse.gef.Handle;
import org.eclipse.gef.handles.AbstractHandle;
import org.eclipse.gef.handles.MoveHandleLocator;
import org.eclipse.gef.handles.NonResizableHandleKit;
import org.eclipse.gef.tools.SelectEditPartTracker;

import com.bitctrl.lib.eclipse.gef.editpolicies.OnlySelectionEditPolicy;

import de.bsvrz.buv.plugin.dobj.editparts.DoModelEditPart;
import de.bsvrz.buv.plugin.dobj.editparts.LinieEditPart;

/**
 * Seletionspolicy, die allen Linien (Editparts vom Typ {@link LinieEditPart})
 * ein Selektionshandle zuordnet, das eine rote Linie ueber der referenzierten
 * Linie als Markierung zeichnet.
 *
 * @author BitCtrl Systems GmbH, thierfelder
 */
public class MqAlsLinieSelectionEditPolicy extends OnlySelectionEditPolicy {

	@Override
	protected List<?> createSelectionHandles() {
		final List<Handle> list = new ArrayList<>();

		if (getHost() instanceof LinieEditPart) {
			final LinieEditPart<?, ?> lep = (LinieEditPart<?, ?>) getHost();
			list.add(new LinieHandle(lep));
		} else {
			NonResizableHandleKit.addMoveHandle(getHost(), list, new SelectEditPartTracker(getHost()), Cursors.ARROW);
			NonResizableHandleKit.addCornerHandles(getHost(), list, new SelectEditPartTracker(getHost()),
					Cursors.ARROW);
		}

		return list;
	}

	/**
	 * Handle zur Markierunng von Strassensegmenten bzw. Linien mit einem roten
	 * Strich.
	 *
	 * @author BitCtrl Systems GmbH, thierfelder
	 */
	private static final class LinieHandle extends AbstractHandle {

		private LinieHandle(final LinieEditPart<?, ?> owner) {
			setOwner(owner);
			setLocator(new MoveHandleLocator(owner.getFigure()));
			setCursor(Cursors.ARROW);
		}

		@Override
		public void paintFigure(final Graphics g) {
			if (getOwner() != null && getOwner().getFigure() != null && getOwner() instanceof DoModelEditPart<?, ?>) {
				final double zoom = ((DoModelEditPart<?, ?>) getOwner()).getZoomManager().getZoom();
				final PointList originalPoints = ((LinieEditPart<?, ?>) getOwner()).getPoints();
				final PointList scaledPoints = new PointList();
				for (int i = 0; i < originalPoints.size(); i++) {
					scaledPoints.addPoint(new PrecisionPoint(originalPoints.getPoint(i).x * zoom,
							originalPoints.getPoint(i).y * zoom));
				}
				g.setLineWidth(3);
				g.setForegroundColor(ColorConstants.red);
				g.drawPolyline(scaledPoints);
			}
		}

		@Override
		protected DragTracker createDragTracker() {
			return new SelectEditPartTracker(getOwner());
		}

	}

}
