/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.dobj.figures;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.file.Files;

import org.eclipse.core.runtime.SafeRunner;
import org.eclipse.draw2d.Figure;
import org.eclipse.draw2d.Graphics;
import org.eclipse.draw2d.ImageFigure;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.PrecisionDimension;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.ResourceManager;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;

import com.bitctrl.lib.eclipse.draw2d.svg.SVGCache;
import com.bitctrl.lib.eclipse.draw2d.svg.SVGFigure;

import de.bsvrz.buv.plugin.dobj.DoFigure;
import de.bsvrz.buv.plugin.dobj.decorator.ZoomVerhaltenFixFigure;
import de.bsvrz.buv.plugin.dobj.internal.DObjPlugin;
import de.bsvrz.buv.plugin.dobj.util.BildManager;
import de.bsvrz.sys.funclib.bitctrl.modell.att.Feld;
import de.bsvrz.sys.funclib.bitctrl.modell.bitctrlallgemein.objekte.Bild;
import de.bsvrz.sys.funclib.bitctrl.modell.metamodellglobal.attribute.AttByte_JavaKeyword;

/**
 * Bild - Figur, die sowohl SVG, als auch normal binary Bilder (png,bmp,jpg)
 * darstellen kann.
 *
 * @author BitCtrl Systems GmbH, Hoesel
 *
 */
public class DoBildFigure extends Figure implements DoFigure, ZoomVerhaltenFixFigure {

	private Bild bild;
	private Image image;
	private boolean zoomVerhaltenFix;
	private int hoehe;
	private int breite;
	private double zoom;
	private Point hotspot = new Point();
	private double vonZoom;
	private double bisZoom;

	private final ResourceManager resourceManager;

	/**
	 * Konstruktor für eine Bild - Figur, die sowohl SVG, als auch normal binary
	 * Bilder (png,bmp,jpg) darstellen kann.
	 *
	 * @param resourceManager
	 *            für die Verwaltung der {@link Image}s.
	 */
	public DoBildFigure(final ResourceManager resourceManager) {
		super();
		this.resourceManager = resourceManager;
	}

	/**
	 * @return the bild
	 */
	public final Bild getBild() {
		return bild;
	}

	/**
	 * Erneutes skalieren und "zeichnen" des Bildes auf eine interne
	 * {@link Figure}.
	 */
	private void refreshBild() {
		if (image != null && !image.isDisposed()) {
			image.dispose();
			image = null;
		}

		setVisible(vonZoom <= zoom && zoom <= bisZoom);

		if (zoomVerhaltenFix) {
			final Rectangle bound = new Rectangle(hotspot, new PrecisionDimension(breite, hoehe).getScaled(1 / zoom));
			setBounds(bound);
		} else {
			final Rectangle bound = new Rectangle(hotspot, new PrecisionDimension(breite, hoehe));
			setBounds(bound);
		}

		if (bild != null) {
			if ("image/svg+xml".equalsIgnoreCase(bild.getKdBild().getDatum().getMIMEType())) {

				final Feld<AttByte_JavaKeyword> bilddaten = bild.getKdBild().getDatum().getBilddaten();
				// TODO:u.U. kann man hier das Array noch Cachen
				final byte[] buffer = new byte[bilddaten.size()];
				for (int i = 0; i < bilddaten.size(); ++i) {
					buffer[i] = bilddaten.get(i).byteValue();
				}

				final File file = DObjPlugin.getDefault().getBundle().getBundleContext()
						.getDataFile(bild.getPid() + System.currentTimeMillis() + ".tmp");
				try (FileOutputStream fout = new FileOutputStream(file)) {
					fout.write(buffer);
					fout.flush();
					final SVGCache svg = new SVGCache();
					svg.setURI(file.toURI().toString(), true);
					removeAll();
					final SVGFigure svgFigure = new SVGFigure();
					svgFigure.setSize(breite, hoehe);
					svgFigure.setSvgCache(svg);
					add(svgFigure);

				} catch (final IOException ex) {
					DObjPlugin.getDefault().getLog().error("Laden eines SVG Hintergrundbildes ist fehlgeschlagen.", ex);
				} finally {
					SafeRunner.run(() -> Files.deleteIfExists(file.toPath()));
				}
			} else {

				final ImageDescriptor desc = BildManager.INSTANCE.getImageDescriptor(bild);

				try {
					final ImageData data = desc.getImageData().scaledTo(breite, hoehe);
					image = resourceManager.createImageWithDefault(ImageDescriptor.createFromImageData(data));
				} catch (final NegativeArraySizeException ex) {
					// das resultierende Bild ist zu groß
					image = resourceManager.create(desc);
				}

				removeAll();
				final ImageFigure imgFigure = new ImageFigure(image);
				imgFigure.setSize(breite, hoehe);
				add(imgFigure);
			}
		}
	}

	@Override
	protected final void paintChildren(final Graphics graphics) {
		graphics.setAdvanced(true);
		graphics.setAntialias(SWT.ON);
		if (zoom != 1.0 && isZoomVerhaltenFix()) {
			graphics.scale(1 / zoom);
			graphics.pushState();
			super.paintChildren(graphics);
			graphics.popState();
		} else {
			super.paintChildren(graphics);
		}
	}

	/**
	 * @param bild
	 *            the bild to set
	 */
	public final void setBild(final Bild bild) {
		if (this.bild != null && this.bild.equals(bild) || this.bild == null && bild == null) {
			return;
		}
		this.bild = bild;
		refreshBild();
	}

	@Override
	public void handleZoomChanged(final double newZoom) {
		if (zoom == newZoom) {
			return;
		}
		zoom = newZoom;
		refreshBild();
	}

	@Override
	public void setSize(final int w, final int h) {
		hoehe = h;
		breite = w;
		refreshBild();
	}

	@Override
	public boolean isZoomVerhaltenFix() {
		return zoomVerhaltenFix;
	}

	@Override
	public void setZoomVerhaltenFix(final boolean zoomVerhaltenFix) {
		if (this.zoomVerhaltenFix != zoomVerhaltenFix) {
			this.zoomVerhaltenFix = zoomVerhaltenFix;
			refreshBild();
		}

	}

	@Override
	public Point getHotspot() {
		return hotspot;
	}

	@Override
	public void setHotspot(final Point hotspot) {
		if (this.hotspot != null && this.hotspot.equals(hotspot)) {
			return;
		}

		this.hotspot = hotspot;

		refreshBild();

	}

	@Override
	public void setSichtbareZoomStufe(final double vonZoom, final double bisZoom) {
		if (this.vonZoom == vonZoom && this.bisZoom == bisZoom) {
			return;
		}

		this.vonZoom = vonZoom;
		this.bisZoom = bisZoom;

		refreshBild();
	}

	@Override
	protected boolean useLocalCoordinates() {
		return true;
	}
}
