/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.dobj.figures;

import org.eclipse.draw2d.Figure;
import org.eclipse.draw2d.Graphics;
import org.eclipse.draw2d.ScaledGraphics;
import org.eclipse.draw2d.TextUtilities;
import org.eclipse.draw2d.geometry.Dimension;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.swt.graphics.Font;

/**
 * Text - Figure, die ihre Größe dem übergebenen Text und der Schriftart
 * anpasst. Die Figure unterstützt zwei Zoomverhalten: fix und dynamisch.
 *
 * @author BitCtrl Systems GmbH, Hoesel
 *
 */
public class DoTextFigure extends Figure {

	private boolean zoomVerhaltenFix = true;
	private double zoom = 1.0;
	private String text = "";
	private Dimension calculateTextSize = new Dimension();

	/**
	 * Flag ob das Zoomverhalten fix oder dynamisch sein soll.
	 *
	 * @return <code>true</code> für fix und <code>false</code> für dynamisches
	 *         Zoomverhalten.
	 */
	public boolean isZoomVerhaltenFix() {
		return zoomVerhaltenFix;
	}

	/**
	 * @see #isZoomVerhaltenFix()
	 */
	public void setZoomVerhaltenFix(final boolean zoomVerhaltenFix) {
		if (this.zoomVerhaltenFix == zoomVerhaltenFix) {
			return;
		}

		this.zoomVerhaltenFix = zoomVerhaltenFix;
		updateSize();
	}

	/**
	 * {@link #setText(String)}
	 */
	public String getText() {
		return text;
	}

	/**
	 * Legt den anzuzeigenden Text fest.
	 */
	public void setText(final String text) {
		if (this.text.equals(text)) {
			return;
		}

		this.text = text != null ? text : "";

		updateSize();
		repaint();
	}

	private void updateSize() {
		if (getFont() == null) {
			return;
		}

		calculateTextSize = TextUtilities.INSTANCE.getTextExtents(getText(), getFont());

		if (zoomVerhaltenFix) {
			final Dimension size = calculateTextSize;
			setSize((int) (size.width / zoom), (int) (size.height / zoom));
		} else {
			setSize(calculateTextSize);
		}
	}

	@Override
	public void setFont(final Font f) {
		super.setFont(f);
		updateSize();
		repaint();
	}

	/**
	 * Muss bei Änderung der Zoomstufe aufgerufen werden, damit die Figure sich
	 * anpassen kann.
	 */
	public void handleZoomChanged(final double newZoom) {
		if (zoom == newZoom) {
			return;
		}

		zoom = newZoom;

		updateSize();
	}

	@Override
	protected void paintFigure(final Graphics graphics) {
		super.paintFigure(graphics);

		if (isZoomVerhaltenFix() && graphics instanceof ScaledGraphics) {
			final Point p = getLocation();
			graphics.translate(p);
			graphics.scale(1 / zoom);
			graphics.drawText(getText(), 0, 0);
			graphics.translate(p.getNegated());
		} else {
			graphics.drawText(getText(), getLocation());
		}
	}

}
