/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dobj.graph;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import com.bitctrl.graph.AbstractGraph;
import com.bitctrl.graph.Bogen;
import com.bitctrl.graph.Knoten;

import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.konfigurationsdaten.KdAeusseresStrassenSegment.Daten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.AeusseresStrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.Netz;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.NetzBestandTeil;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.StrassenKnoten;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Repräsentiert einen Graph aus Straßenknoten und äußeren Straßensegmenten.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class NetzGraph extends AbstractGraph {

	private static final Debug LOGGER = Debug.getLogger();
	private final Netz netz;
	private final Map<StrassenKnoten, Knoten> knotenListe = new HashMap<>();
	private final Map<AeusseresStrassenSegment, Bogen> bogenListe = new HashMap<>();

	/**
	 * Erzeugt einen Netzgraph.
	 */
	public NetzGraph(final Netz netz) {
		this.netz = netz;
		init(netz);
	}

	/**
	 * Gibt das gekapselte netz zurück.
	 */
	public Netz getNetz() {
		return netz;
	}

	private void init(final Netz n) {

		for (final NetzBestandTeil nbt : n.getNetzBestandTeile()) {
			if (nbt instanceof Netz) {
				// Verschachteltes Netz analysieren
				init((Netz) nbt);
			} else if (nbt instanceof final AeusseresStrassenSegment aess) {
				final Daten datum = aess.getKdAeusseresStrassenSegment().getDatum();
				if (datum != null) {
					// Anfangsknoten des neuen Bogens registrieren
					final StrassenKnoten vonKnoten = datum.getVonKnoten();
					NetzKnoten anfangsKnoten = null;
					if (vonKnoten != null) {
						if (!knotenListe.containsKey(vonKnoten)) {
							anfangsKnoten = new NetzKnoten(vonKnoten);
							knotenListe.put(vonKnoten, anfangsKnoten);
						} else {
							anfangsKnoten = (NetzKnoten) knotenListe.get(vonKnoten);
						}
					}

					// Endknoten des neuen Bogens registrieren
					final StrassenKnoten nachKnoten = datum.getNachKnoten();
					NetzKnoten endKnoten = null;
					if (nachKnoten != null) {
						if (!knotenListe.containsKey(nachKnoten)) {
							endKnoten = new NetzKnoten(nachKnoten);
							knotenListe.put(nachKnoten, endKnoten);
						} else {
							endKnoten = (NetzKnoten) knotenListe.get(nachKnoten);
						}
					}

					// Neuen Bogen registrieren
					bogenListe.put(aess, new NetzBogen(aess, anfangsKnoten, endKnoten));
				} else {
					LOGGER.warning(
							"Äußeres Straßensegment ist nicht vollständig konfiguriert und wird deshalb ignoriert:",
							aess);
				}
			} else {
				// Unbekanter Netzbestandteil gefunden
				throw new IllegalStateException("Unbekanntes NetzBestandTeil: " + nbt);
			}
		}

		for (final Knoten knoten : getKnoten()) {
			((NetzKnoten) knoten).init(this);
		}

	}

	/**
	 * Gibt den Netzknoten zu Straßenknoten zurück.
	 *
	 * @param strassenKnoten
	 *            der Straßenknoten zu dem der Netzknoten gesucht wird.
	 * @return der Netzknoten oder <code>null</code>, wenn der Straßenknoten
	 *         nicht zum Netz gehört.
	 */
	public NetzKnoten getKnoten(final StrassenKnoten strassenKnoten) {
		return (NetzKnoten) knotenListe.get(strassenKnoten);
	}

	@Override
	public Set<Knoten> getKnoten() {
		return Collections.unmodifiableSet(new HashSet<>(knotenListe.values()));
	}

	/**
	 * Gibt den Netzbogen zu einem äußeren Straßensegment zurück.
	 *
	 * @param aeusseresStrassenSegment
	 *            das äußere Straßensegment zu dem der Netzbogen gesucht wird.
	 * @return der Netzbogen oder <code>null</code>, wenn das äußere
	 *         Straßensegment nicht zum Netz gehört.
	 */
	public NetzBogen getBogen(final AeusseresStrassenSegment aeusseresStrassenSegment) {
		return (NetzBogen) bogenListe.get(aeusseresStrassenSegment);
	}

	@Override
	public Collection<Bogen> getBoegen() {
		return Collections.unmodifiableSet(new HashSet<>(bogenListe.values()));
	}

}
