/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dobj.graph;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import com.bitctrl.graph.AbstractKnoten;
import com.bitctrl.graph.Bogen;

import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.konfigurationsdaten.KdInneresStrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.AeusseresStrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.InneresStrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.StrassenKnoten;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Repräsentiert einen Knoten (Straßenknoten) im Netzgraph.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class NetzKnoten extends AbstractKnoten {

	private record NetzKnotenKey(AeusseresStrassenSegment von, AeusseresStrassenSegment nach) {
	}

	private static final Debug LOGGER = Debug.getLogger();
	private final StrassenKnoten strassenKnoten;
	private final Set<Bogen> eingangsBogenListe = new HashSet<>();
	private final Set<Bogen> ausgangsBogenListe = new HashSet<>();
	private final Set<Bogen> inzidentBogenListe = new HashSet<>();
	private final Map<NetzKnotenKey, InneresStrassenSegment> innereStrassenSegmente = new HashMap<>();

	/**
	 * Initialisiert den Netzknoten.
	 */
	NetzKnoten(final StrassenKnoten strassenKnoten) {
		this.strassenKnoten = strassenKnoten;
	}

	/**
	 * Gibt den gekapselten Straßenknoten zurück.
	 */
	public StrassenKnoten getStrassenKnoten() {
		return strassenKnoten;
	}

	/**
	 * Gibt das innere Straßensegment zurück, welches zwei äußere
	 * Straßensegmente verbindet.
	 *
	 * @param vonStrassenSegment
	 *            das Straßensegment, welches zu diesem Straßenknoten hinführt.
	 * @param nachStrassenSegment
	 *            das Straßensegment, welches von diesem Straßenknoten wegführt.
	 * @return das innere Straßensegment, welches die beiden äußeren verbindet
	 *         oder <code>null</code>, wenn es keines gibt.
	 */
	public InneresStrassenSegment getInneresStrassenSegment(final AeusseresStrassenSegment vonStrassenSegment,
			final AeusseresStrassenSegment nachStrassenSegment) {
		return innereStrassenSegmente.get(new NetzKnotenKey(vonStrassenSegment, nachStrassenSegment));
	}

	void init(final NetzGraph graph) {
		eingangsBogenListe.clear();
		ausgangsBogenListe.clear();
		inzidentBogenListe.clear();

		for (final InneresStrassenSegment iss : strassenKnoten.getInnereStrassenSegmente()) {
			final KdInneresStrassenSegment.Daten issDatum = iss.getKdInneresStrassenSegment().getDatum();
			if (issDatum == null) {
				LOGGER.warning("Inneres Straßensegment ist nicht vollständig konfiguriert und wird deshalb ignoriert:",
						iss);
				continue;
			}

			final AeusseresStrassenSegment nachStrassenSegment = issDatum.getNachStrassenSegment();
			if (nachStrassenSegment != null) {
				final NetzBogen bogen = graph.getBogen(nachStrassenSegment);
				if (bogen == null) {
					// System.err
					// .println("Ausgangsbogen ist nicht im Graph enthalten: "
					// + nachStrassenSegment);
				} else if (equals(bogen.getAnfangsKnoten())) {
					ausgangsBogenListe.add(bogen);
					inzidentBogenListe.add(bogen);
				} else {
					// System.err
					// .println("Anfangsknoten des Ausgangsbogen ist falsch: "
					// + bogen);
				}
			}

			final AeusseresStrassenSegment vonStrassenSegment = issDatum.getVonStrassenSegment();
			if (vonStrassenSegment != null) {
				final NetzBogen bogen = graph.getBogen(vonStrassenSegment);
				if (bogen == null) {
					// System.err
					// .println("Eingangsbogen ist nicht im Graph enthalten: "
					// + nachStrassenSegment);
				} else if (equals(bogen.getEndKnoten())) {
					eingangsBogenListe.add(bogen);
					inzidentBogenListe.add(bogen);
				} else {
					System.err.println("Endknoten des Eingangsbogen ist falsch: " + bogen);
				}
			}

			if (vonStrassenSegment != null && nachStrassenSegment != null) {
				innereStrassenSegmente.put(new NetzKnotenKey(vonStrassenSegment, nachStrassenSegment), iss);
			}

			if (vonStrassenSegment == null && nachStrassenSegment == null) {
				System.err.println("Knoten ohne inzidente Bögen: " + this);
			}
		}
	}

	@Override
	public Iterable<Bogen> ausgangsBogenIterator() {
		return ausgangsBogenListe;
	}

	@Override
	public Iterable<Bogen> eingangsBogenIterator() {
		return eingangsBogenListe;
	}

	@Override
	public Iterable<Bogen> inzidentBogenIterator() {
		return inzidentBogenListe;
	}

	@Override
	public void setStuetzBogen(final Bogen stuetzBogen) {
		if (stuetzBogen == null || stuetzBogen == WURZEL_BOGEN) {
			super.setStuetzBogen(stuetzBogen);
			return;
		}

		if (!equals(stuetzBogen.getEndKnoten())) {
			throw new IllegalArgumentException("Der Stützbogen muss zu diesem Knotenführen.");
		}

		final AeusseresStrassenSegment nachStrassenSegment = ((NetzBogen) stuetzBogen).getAeusseresStrassenSegment();

		final NetzKnoten lastKnoten = (NetzKnoten) stuetzBogen.getAnfangsKnoten();
		final Bogen lastStuetzBogen = lastKnoten.getStuetzBogen();
		if (lastStuetzBogen == null || lastStuetzBogen == WURZEL_BOGEN) {
			super.setStuetzBogen(stuetzBogen);
			return;
		}

		final AeusseresStrassenSegment vonStrassenSegment = ((NetzBogen) lastStuetzBogen).getAeusseresStrassenSegment();

		if (lastKnoten.getInneresStrassenSegment(vonStrassenSegment, nachStrassenSegment) != null) {
			super.setStuetzBogen(stuetzBogen);
		} else {
			System.err.println("Ungültiger Stützbogen wird ignoriert.");
		}
	}

	@Override
	public boolean equals(final Object obj) {
		if (this == obj) {
			return true;
		} else if (obj instanceof final NetzKnoten other) {
			return strassenKnoten.equals(other.strassenKnoten);
		}
		return false;
	}

	@Override
	public int hashCode() {
		return strassenKnoten.hashCode();
	}

	@Override
	public String toString() {
		return strassenKnoten.toString();
	}

}
