/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dobj.internal;

import org.eclipse.core.runtime.IAdapterFactory;
import org.eclipse.gef.EditPart;

import de.bsvrz.buv.plugin.dobj.model.BitCtrlDoModel;
import de.bsvrz.buv.plugin.dobj.model.DoModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.bitctrl.modell.SystemObjekt;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Adapter Factory für Darstellungsobjekte.
 *
 * Folgende Adapter stehen zur Verfügung:
 *
 * <ul>
 * <li>{@link EditPart} -> {@link DoModel}</li>
 * <li>{@link EditPart} -> {@link SystemObjekt}</li>
 * <li>{@link EditPart} -> {@link SystemObject}</li>
 * </ul>
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class AdapterFactory implements IAdapterFactory {

	private static final Debug LOGGER = Debug.getLogger();
	private Class<?>[] adapterList;

	/**
	 * Schreibt eine Meldung über die zur Verfügung gestellten Adapter ins
	 * Datenverteiler-Log unter dem Level <code>fine</code>.
	 */
	public AdapterFactory() {
		LOGGER.fine("Adapter initialisiert für " + EditPart.class + " nach",
				new Object[] { SystemObject.class, SystemObjekt.class, DoModel.class });
	}

	@Override
	public Object getAdapter(final Object adaptableObject, final Class adapterType) {
		final Object model = ((EditPart) adaptableObject).getModel();

		if (model instanceof BitCtrlDoModel) {
			final BitCtrlDoModel<?> doModel = (BitCtrlDoModel<?>) model;
			if (SystemObjekt.class == adapterType) {
				return doModel.getSystemObjekt();
			}
		}

		// Datenverteiler-Modell
		if (model instanceof final DoModel doModel) {
			if (DoModel.class == adapterType) {
				return model;
			} else if (SystemObject.class == adapterType) {
				return doModel.getSystemObject();
			}
		}

		return null;
	}

	@Override
	public Class<?>[] getAdapterList() {
		if (adapterList == null) {
			adapterList = new Class[] { SystemObject.class, SystemObjekt.class, DoModel.class };
		}
		return adapterList.clone();
	}

}
