/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dobj.internal;

import java.util.ArrayList;
import java.util.Collection;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.RegistryFactory;

import com.bitctrl.lib.eclipse.emf.util.EmfUtil;

import de.bsvrz.buv.plugin.dobj.model.ConfiguratedDoTyp;
import de.bsvrz.buv.plugin.dobj.model.DobjFactory;
import de.bsvrz.dav.daf.main.config.SystemObjectType;

/**
 * Vereinfacht den Zugriff auf den Extension Point {@value #EXTENSION_POINT_ID}.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 */
public final class DarstellungExtensionPoint {

	/** Die ID des Extension Points. */
	public static final String EXTENSION_POINT_ID = "de.bsvrz.buv.plugin.dobj.darstellung";

	/** Die ID des entsprechenden Elements. */
	public static final String ELEMENT_DARSTELLUNGS_OBJEKT_TYP = "darstellungsObjektTyp";

	/** Der Name des entsprechenden Attributs. */
	public static final String ATT_ID = "id";

	/** Der Name des entsprechenden Attributs. */
	public static final String ATT_NAME = "name";

	/** Der Name des entsprechenden Attributs. */
	public static final String ATT_DESCRIPTION = "description";

	/** Der Name des entsprechenden Attributs. */
	public static final String ATT_SYSTEM_OBJEKT_TYP = "systemObjektTyp";

	/** Der Name des entsprechenden Attributs. */
	public static final String ATT_MODEL_CLASS = "modelClass";

	/** Der Name des entsprechenden Attributs. */
	public static final String ATT_CLASS = "class";

	/** Der Name des entsprechenden Attributs. */
	public static final String ATT_CONTROLLER_CLASS = "controllerClass";

	/** Der Name des entsprechenden Attributs. */
	public static final String ATT_ICON = "icon";

	/**
	 * Flag, ob der Darstellungsobjekt nur in der maßstäblichen Darstellung
	 * verwendet werden kann.
	 */
	public static final String ATT_NUR_MASSSTAEBLICHE_DARSTELLUNG = "nurMassstaeblicheDarstellung";

	/**
	 * Gibt den Darstellungsobjekttyp zu einer Extension-ID zurück.
	 *
	 * @param id
	 *            die ID der gesuchten Extension.
	 * @return der Darstellungsobjekttyp zur Extension-ID oder <code>null</code>
	 *         , wenn kein Darstellungsobjekttyp zu der Extension-ID existiert.
	 */
	public static ConfiguratedDoTyp getDoTyp(final String id) {
		if (id == null) {
			throw new IllegalArgumentException("Argument darf nicht null sein.");
		}

		final IExtensionRegistry registry = RegistryFactory.getRegistry();
		final IConfigurationElement[] elements = registry.getConfigurationElementsFor(EXTENSION_POINT_ID);
		for (final IConfigurationElement e : elements) {
			if (ELEMENT_DARSTELLUNGS_OBJEKT_TYP.equals(e.getName())) {
				final String att = e.getAttribute("id");
				if (att != null && att.equals(id)) {
					return createDoTyp(e);
				}
			}
		}

		return null;
	}

	private static ConfiguratedDoTyp createDoTyp(final IConfigurationElement e) {
		final ConfiguratedDoTyp doTyp;
		final String clazz = e.getAttribute(ATT_CLASS);
		if (clazz != null) {
			final int index = clazz.lastIndexOf('/');
			doTyp = (ConfiguratedDoTyp) EmfUtil.createEObject(clazz.substring(0, index), clazz.substring(index + 1));
		} else {
			// Fallback
			doTyp = DobjFactory.eINSTANCE.createConfiguratedDoTyp();
		}

		doTyp.setConfigurationElement(e);
		return doTyp;
	}

	/**
	 * Gibt alle Darstellungsobjekttypen zurück, die zur Verfügung stehen.
	 *
	 * @return alle bekannten Darstellungsobjekttypen.
	 */
	public static Collection<ConfiguratedDoTyp> getDoTypen() {
		final Collection<ConfiguratedDoTyp> typen = new ArrayList<>();

		final IExtensionRegistry extension = Platform.getExtensionRegistry();
		final IConfigurationElement[] elements = extension.getConfigurationElementsFor(EXTENSION_POINT_ID);
		for (final IConfigurationElement e : elements) {
			if (!ELEMENT_DARSTELLUNGS_OBJEKT_TYP.equals(e.getName())) {
				continue;
			}

			final ConfiguratedDoTyp doTyp = createDoTyp(e);
			final String systemObjektTypPid = e.getAttribute(ATT_SYSTEM_OBJEKT_TYP);

			if (isEmptyOrNull(systemObjektTypPid) || doTyp.getSystemObjectType() != null) {
				typen.add(doTyp);
			}
		}

		return typen;
	}

	private static boolean isEmptyOrNull(final String systemObjektTypPid) {
		if (systemObjektTypPid == null || systemObjektTypPid.trim().isEmpty()) {
			return true;
		}

		return false;
	}

	/**
	 * Gibt alle Darstellungsobjekttypen zurück, die für einen bestimmten
	 * Systemobjekttyp zur Verfügung stehen. Es werden die
	 * Darstellungsobjekttypen zurückgegegeben, die exakt den angegebenen
	 * Systemobjekttyp besitzen.
	 *
	 * @param systemObjectType
	 *            der Systemobjekttyp für den die passenden Darstellungsobjekten
	 *            gesucht sind.
	 * @return alle bekannten Darstellungsobjekttypen zum übergebenen
	 *         Systemobjekttyp.
	 */
	public static Collection<ConfiguratedDoTyp> getDoTypenWithExactMatch(final SystemObjectType systemObjectType) {
		if (systemObjectType == null) {
			throw new IllegalArgumentException("Argument darf nicht null sein.");
		}

		final Collection<ConfiguratedDoTyp> filteredTypen = new ArrayList<>();

		for (final ConfiguratedDoTyp doTyp : getDoTypen()) {
			if (systemObjectType.equals(doTyp.getSystemObjectType())) {
				filteredTypen.add(doTyp);
			}
		}

		return filteredTypen;
	}

	/**
	 * Gibt alle Darstellungsobjekttypen zurück, die für einen bestimmten
	 * Systemobjekttyp zur Verfügung stehen. Es werden die
	 * Darstellungsobjekttypen zurückgegegeben, die angegebenen Systemobjekttyp
	 * oder einem seiner Supertypen besitzen.
	 *
	 * @param systemObjectType
	 *            der Systemobjekttyp für den die passenden Darstellungsobjekten
	 *            gesucht sind.
	 * @return alle bekannten Darstellungsobjekttypen zum übergebenen
	 *         Systemobjekttyp.
	 */
	public static Collection<ConfiguratedDoTyp> getDoTypen(final SystemObjectType systemObjectType) {
		if (systemObjectType == null) {
			throw new IllegalArgumentException("Argument darf nicht null sein.");
		}

		final Collection<ConfiguratedDoTyp> filteredTypen = new ArrayList<>();

		for (final ConfiguratedDoTyp doTyp : getDoTypen()) {
			if (systemObjectType.equals(doTyp.getSystemObjectType())
					|| systemObjectType.getSuperTypes().contains(doTyp.getSystemObjectType())) {
				filteredTypen.add(doTyp);
			}
		}

		return filteredTypen;
	}

	private DarstellungExtensionPoint() {
		// utility class
	}

}
