/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.dobj.kollision.travers;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;

import org.eclipse.core.runtime.Assert;
import org.eclipse.draw2d.geometry.Point;

/**
 * Traversiert eine Menge von Punkten im Draw2D. Eine Instanz dieser Klasse wird
 * immer ueber einen bestimmten Punkt (z.B. den Startpunkt der Traversierung
 * identifiziert).
 *
 * @author BitCtrl Systems GmbH, thierfelder
 *
 */
class PunktTraverser implements IPunktTraverser, Comparable<PunktTraverser> {

	/**
	 * Eindeutiger Identifikationspunkt.
	 */
	private final Point idPunkt;

	/**
	 * Punktliste die traversiert werden soll.
	 */
	protected final List<Point> punktListe = new ArrayList<>();

	/**
	 * Standardkonstruktor.
	 *
	 * @param idPunkt
	 *            Eindeutiger Identifikationspunkt (identifiziert dieses Objekt
	 *            eindeutig, z.B. Startpunkt der Traversierung). Muss ungleich
	 *            <code>null</code> sein.
	 */
	PunktTraverser(final Point idPunkt) {
		Assert.isNotNull(idPunkt, "idPunkt");
		this.idPunkt = idPunkt.getCopy();
	}

	@Override
	public Iterator<Point> iterator() {
		if (!punktListe.isEmpty()) {
			return punktListe.iterator();
		}

		return Collections.singletonList(idPunkt).iterator();
	}

	@Override
	public int compareTo(final PunktTraverser that) {
		final Integer thisX = idPunkt.x;
		final Integer thatX = that.idPunkt.x;

		final int xCompareResult = thisX.compareTo(thatX);
		if (xCompareResult == 0) {
			final Integer thisY = idPunkt.y;
			final Integer thatY = that.idPunkt.y;
			return thisY.compareTo(thatY);
		}

		return xCompareResult;
	}

	@Override
	public int hashCode() {
		return Objects.hash(idPunkt, punktListe);
	}

	@Override
	public boolean equals(final Object obj) {
		if (this == obj) {
			return true;
		}
		if (obj == null || getClass() != obj.getClass()) {
			return false;
		}
		final PunktTraverser other = (PunktTraverser) obj;
		return Objects.equals(idPunkt, other.idPunkt) && Objects.equals(punktListe, other.punktListe);
	}

}
