/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.dobj.legende;

import java.util.Collections;
import java.util.List;

import org.eclipse.draw2d.IFigure;
import org.eclipse.draw2d.geometry.Dimension;
import org.eclipse.swt.graphics.Image;

import com.bitctrl.lib.eclipse.draw2d.FigureUtilities2;

import de.bsvrz.buv.plugin.dobj.model.DoTyp;
import de.bsvrz.buv.rw.basislib.legende.ILegendeBaustein;

/**
 * Allgemeiner Legendenbaustein.
 *
 * <p>
 * Zeigt als Text den Namen des {@link DoTyp}s und als Bild ein 16x16 Pixel
 * großes Icon der Figure. Der Legendenbaustein hat keine weiteren
 * Unterbausteine.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class DoLegendeBaustein implements ILegendeBaustein {

	private final DoTyp doTyp;
	private final IFigure figure;
	private final Dimension figureSize;

	private Image image;

	/**
	 * Legt den {@link DoTyp} und die Figure für die Darstellung in der Legende
	 * fest.
	 */
	public DoLegendeBaustein(final DoTyp doTyp, final IFigure figure) {
		this(doTyp, figure, null);
	}

	/**
	 * Legt den {@link DoTyp} und die Figure für die Darstellung in der Legende
	 * fest.
	 */
	public DoLegendeBaustein(final DoTyp doTyp, final IFigure figure, final Dimension figureSize) {
		this.doTyp = doTyp;
		this.figure = figure;
		this.figureSize = figureSize;
	}

	/**
	 * Gibt den Darstellungsobjekttyp zurück, den der Legendenbaustein
	 * darstellt.
	 */
	protected final DoTyp getDoTyp() {
		return doTyp;
	}

	/**
	 * Gibt die Figure zurück, mit der der Darstellungsobjekttyp dargestellt
	 * wird.
	 */
	protected final IFigure getFigure() {
		return figure;
	}

	/**
	 * Die Defaultimplementierung gibt eine leere Liste zurück.
	 */
	@Override
	public List<ILegendeBaustein> getBausteine() {
		return Collections.emptyList();
	}

	/**
	 * Erstellt ein 16x16 Pixel großes Icon aus der Figure.
	 */
	@Override
	public final Image getIcon() {
		if (image == null) {
			if (figureSize != null) {
				image = FigureUtilities2.createScaledImage(figure, figureSize, new Dimension(16, 16));
			} else {
				image = FigureUtilities2.createImage(figure, new Dimension(16, 16));
			}
		}

		return image;
	}

	/**
	 * Gibt den Namen des Darstellungsobjekttyps zurück.
	 */
	@Override
	public final String getText() {
		return doTyp.getName();
	}

	/**
	 * Gibt verwendete Resourcen wieder frei. Muss manuell aufgerufen werden,
	 * wenn der Legendenbaustein nicht mehr verwendet wird.
	 */
	public void dispose() {
		if (image != null) {
			image.dispose();
			image = null;
		}

	}

}
