/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dobj.tools;

import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Set;

import org.eclipse.draw2d.Cursors;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.EditPartViewer;
import org.eclipse.gef.GraphicalViewer;
import org.eclipse.gef.tools.SelectEditPartTracker;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.Shell;

import com.bitctrl.lib.eclipse.emf.util.EmfUtil;

import de.bsvrz.buv.plugin.dobj.model.DoModel;

/**
 * Ermoeglicht die Auswahl von uebereinanderliegenden Objekten.
 *
 * @author BitCtrl Systems GmbH, thierfelder
 */
class CrowdedEditPartsTracker extends SelectEditPartTracker {

	/**
	 * Zum Sortieren von {@link EditPart}s (alphabetisch nach Namen).
	 */
	private static final Comparator<EditPart> EDIT_PART_COMPARATOR = (o1, o2) -> {
		if (o1 != null && o2 != null && o1.getModel() != null && o2.getModel() != null) {
			if (o1.getModel() instanceof EObject && o2.getModel() instanceof EObject) {
				return EmfUtil.getText((EObject) o1.getModel()).compareTo(EmfUtil.getText((EObject) o2.getModel()));
			}
			return o1.getModel().toString().compareTo(o2.getModel().toString());
		}
		return 0;
	};

	private final Set<EditPart> moeglicheSelektionen;

	private EditPartViewer activationViewer;

	/**
	 * Konstruktor.
	 *
	 * @param sourceEditPart
	 *            Source-{@link EditPart}
	 * @param moeglicheSelektionen
	 *            die EditParts die durch die Selektion von Source-
	 *            {@link EditPart} auch gemeint sein koennten.
	 */
	CrowdedEditPartsTracker(final EditPart sourceEditPart, final Set<EditPart> moeglicheSelektionen) {
		super(sourceEditPart);
		this.moeglicheSelektionen = moeglicheSelektionen;
		setDisabledCursor(Cursors.NO);
	}

	@Override
	public void activate() {
		activationViewer = getCurrentViewer();
		if (activationViewer != null) {
			trySelektiereEinfachesObjekt((GraphicalViewer) activationViewer, moeglicheSelektionen);
		}
	}

	@Override
	protected String getDebugName() {
		return "CrowdedEditPartsTracker: " + getCommandName();
	}

	private void trySelektiereEinfachesObjekt(final GraphicalViewer viewer, final Set<EditPart> editPartSet) {
		if (editPartSet == null || editPartSet.isEmpty()) {
			return;
		}

		final Display display = viewer.getControl().getDisplay();
		final Shell shell = new Shell(display);
		final Menu menu = new Menu(shell, SWT.POP_UP);

		final DOSelektionsListener listener = new DOSelektionsListener();

		final List<EditPart> sortedEditPartList = Arrays.asList(editPartSet.toArray(new EditPart[0]));
		Collections.sort(sortedEditPartList, EDIT_PART_COMPARATOR);

		for (final EditPart editPart : sortedEditPartList) {
			if (editPart.isSelectable()) {
				final MenuItem item = new MenuItem(menu, SWT.CHECK);
				item.setData(editPart);
				if (editPart.getModel() instanceof DoModel) {
					final DoModel doModel = (DoModel) editPart.getModel();
					item.setText(EmfUtil.getText(doModel) + "; " + EmfUtil.getText(doModel.getDoTyp()));
				} else {
					item.setText(editPart.getModel().toString());
				}
				item.setSelection(editPart.getSelected() != EditPart.SELECTED_NONE);
				item.addSelectionListener(listener);
			}
		}

		menu.setVisible(true);
	}

	/**
	 * Fuert die Selektion durch.
	 *
	 * @author BitCtrl Systems GmbH, thierfelder
	 */
	private class DOSelektionsListener extends SelectionAdapter {

		@Override
		public void widgetSelected(final SelectionEvent e) {
			if (e.getSource() != null && e.getSource() instanceof MenuItem) {
				final MenuItem item = (MenuItem) e.getSource();
				if (item.getData() != null && item.getData() instanceof EditPart) {
					final EditPart gep = (EditPart) item.getData();
					activationViewer.select(gep);
				}
			}
		}

	}

}
